/******************************************************************************
 * This program is subject to copyright protection in accordance with the
 * applicable law and is also subject to the Developer Download Software
 * License Agreement, ("License Agreement") agreed to by the licensee prior to
 * download and receipt of this program.  It must only be used for the purpose
 * of developing software that is compatible with the CLI handheld computer.
 * It must not, except as permitted in the License Agreement, by any means or
 * in any form be reproduced, distributed or lent.  Moreover, no part of the
 * program may be used, viewed, printed, disassembled or otherwise interfered
 * with in any form, except where allowed by the License Agreement, without the
 * express written consent of the copyright holder.
 *
 * THIS PROGRAM IS FURTHER PROVIDED "AS IS" WITH NO WARRANTY OF ANY KIND AND
 * THE COPYRIGHT HOLDER HEREBY DISCLAIMS ALL WARRANTIES, EXPRESS AND IMPLIED AS
 * TO THE PROGRAM.
 *
 * Copyright (c) 2002-2003 Sony Electronics Inc.
 * Some portions copyright (c) 1999 Palm Computing, Inc. or its subsidiaries.
 * All Rights Reserved.
 *
 * File: MainForm.cpp
 *
 *****************************************************************************/


/** MainForm -- the main Form for the SilkSample project;
  *             this Form supports resizing the virtual silkscreen area and
  *               demonstrates how to reposition widgets appropriately
  */

#include <PalmOS.h>
#include <SonyCLIE.h>

#include "SilkSample.h"
#include "SilkSampleRsc.h"



/* FUNCTION DEFINITIONS ------------------------------------------------ */

/***********************************************************************
 * FUNCTION:    PrvMoveMainFormObjects
 * DESCRIPTION: moves the objects on the Main Form appropriately for
 *                the given change in window bounds
 * PARAMETERS:  IN/OUT frmP - pointer to the Main Form.
 *              dx          - the change in width of the window bounds
 *              dy          - the change in height of the window bounds
 * RETURNS:
 ***********************************************************************/
static void PrvMoveMainFormObjects(FormType* frmP, Coord dx, Coord dy)
{
    if (gSilkState.refNum != sysInvalidRefNum && (dx != 0 || dy != 0))
    {
        RectangleType objBounds;

        // NOTE: be careful to avoid object migration!

        MoveFormObject(frmP, MainToggleFullscreenButton, dx / 2, dy);
        MoveFormObject(frmP, MainDescriptionLabel, dx / 2, 0);
        MoveFormObject(frmP, MainGoto1PushButton, dx, 0);
        MoveFormObject(frmP, MainGoto2PushButton, dx, 0);
        MoveFormObject(frmP, MainGoto3PushButton, dx, 0);

        // resize the Sample Field
        FieldType* fieldP = (FieldType*) GetObjectPtr(frmP, MainSampleField);
        FldGetBounds(fieldP, &objBounds);
        objBounds.extent.x += dx;
        FldSetBounds(fieldP, &objBounds);
        FldRecalculateField(fieldP, true);

        // move the Graffiti Shift indicator
        // (FrmGetObjectIndex does not work on GSI objects)
        UInt16 index = GetGSIIndex(frmP);
        FrmGetObjectBounds(frmP, index, &objBounds);
        FrmSetObjectPosition(frmP, index,
                             objBounds.topLeft.x + dx,
                             objBounds.topLeft.y + dy);
    }
}


/***********************************************************************
 * FUNCTION:    PrvMainFormInit
 * DESCRIPTION: This routine initializes the Main Form.
 * PARAMETERS:  IN/OUT frmP - pointer to the Main Form.
 * RETURNS:
 ***********************************************************************/
static void PrvMainFormInit(FormType* frmP)
{
    // enable silkscreen resizing for this Form
    if (gSilkState.refNum != sysInvalidRefNum)
    {
        if (!gSilkState.resizeEnabled)
        {
            // the silkscreen explicitly must be set to be resizable
            if (EnableSilkResize(gSilkState.refNum, vskResizeVertically | vskResizeHorizontally)
                == errNone)
            {
                gSilkState.resizeEnabled = true;
            }
        }

        // enable the ToggleFullscreen button
        // (in this sample application, the ToggleFullscreen button is
        // enabled only if the Silk library is usable, since it is useless
        // otherwise)
        UInt16 index = FrmGetObjectIndex(frmP, MainToggleFullscreenButton);
        ErrFatalDisplayIf(index == frmInvalidObjectId,
                          "MainToggleFullscreenButton not found");
        FrmShowObject(frmP, index);

        // if resizeAmount.x != 0 || resizeAmount.y != 0, then we've
        // already resized the window and don't need to do it again
        if (gSilkState.resizeAmount.x == 0 && gSilkState.resizeAmount.y == 0)
        {
            ResizeWindowToDisplay(FrmGetWindowHandle(frmP), &gSilkState.resizeAmount.x, &gSilkState.resizeAmount.y);
        }
        PrvMoveMainFormObjects(frmP, gSilkState.resizeAmount.x, gSilkState.resizeAmount.y);

        gSilkState.curResizableFormP = frmP;
        gSilkState.resizeAmount.x = gSilkState.resizeAmount.y = 0;
    }

    // initialize Form selection pushbuttons
    FrmSetControlGroupSelection(frmP, MainGroupID, MainGoto1PushButton);
}


/***********************************************************************
 * FUNCTION:    MainFormHandleEvent
 * DESCRIPTION: This routine is the event handler for the Main Form
 * PARAMETERS:  IN eventP - a pointer to an EventType structure
 * RETURNS:     true if the event has been handled and should not be passed
 *              to a higher level handler.
 ***********************************************************************/
Boolean MainFormHandleEvent(EventType* eventP)
{
    Boolean handled = false;

    FormType* frmP;

    switch (eventP->eType)
    {
        case menuEvent:
            handled = DoMenuCommand(eventP->data.menu.itemID);
            break;

        case frmOpenEvent:
            frmP = FrmGetActiveForm();

            // PrvMainFormInit initializes the Form bounds and widget
            // positions
            PrvMainFormInit(frmP);
            FrmDrawForm(frmP);

            FrmSetFocus(frmP, FrmGetObjectIndex(frmP, MainSampleField));

            handled = true;
            break;

        case frmUpdateEvent:
            // update the Form; reposition widgets and perform custom
            // drawing as necessary
            if (gSilkState.resized)
            {
                frmP = FrmGetActiveForm();

                // if a dialog was active when the silkscreen was resized,
                // the system may have cached the old clip bounds and may
                // not have restored them properly when the dialog was
                // dismissed
                WinResetClip();

                // when the silkscreen is minimized, the formerly occupied
                // area sometimes is not cleared with the Form's fill color
                IndexedColorType formFillColor,
                                 oldBackColor;
                formFillColor = UIColorGetTableEntryIndex(UIFormFill);
                oldBackColor = WinSetBackColor(formFillColor);
                WinEraseWindow();
                WinSetBackColor(oldBackColor);

                PrvMoveMainFormObjects(frmP, gSilkState.resizeAmount.x, gSilkState.resizeAmount.y);

                gSilkState.resized = false;
                gSilkState.resizeAmount.x = gSilkState.resizeAmount.y = 0;
            }

            // this Form does not perform any custom drawing; leave the
            // event unhandled to let the system redraw the Form
            break;

        case winEnterEvent:
            // if we are returning from another window (such as a dialog
            // box), check to see if the Main Form needs updating as a
            // result of resizing the silkscreen area
            //
            // (we must check for winEnter events because modal Forms with
            // the save-behind bit enabled may not generate frmUpdate
            // events)
            if (   eventP->data.winEnter.enterWindow
                   == FrmGetWindowHandle(FrmGetActiveForm())
                && gSilkState.resized)
            {
                FrmUpdateForm(MainForm, frmRedrawUpdateCode);
            }
            break;

        case ctlSelectEvent:
            handled = true;
            switch (eventP->data.ctlSelect.controlID)
            {
                case MainToggleFullscreenButton:
                    if (gSilkState.refNum != sysInvalidRefNum)
                    {
                        static UInt16 lastSilkPos = silkResizeNormal;

                        UInt16 silkPos = GetSilkPos(gSilkState.refNum);
                        if (silkPos != silkResizeMax)
                        {
                            (void) ResizeSilk(gSilkState.refNum, vskResizeNone);
                            lastSilkPos = silkPos;
                        }
                        else
                        {
                            (void) ResizeSilk(gSilkState.refNum, lastSilkPos);
                        }
                    }
                    break;

                case MainGoto1PushButton:
                    break;

                case MainGoto2PushButton:
                    FrmGotoForm(SecondForm);
                    break;

                case MainGoto3PushButton:
                    FrmGotoForm(ThirdForm);
                    break;

                default:
                    handled = false;
                    break;
            }
            break;

        default:
            break;
    }

    return handled;
}
