/******************************************************************************
 * This program is subject to copyright protection in accordance with the
 * applicable law and is also subject to the Developer Download Software
 * License Agreement, ("License Agreement") agreed to by the licensee prior to
 * download and receipt of this program.  It must only be used for the purpose
 * of developing software that is compatible with the CLI handheld computer.
 * It must not, except as permitted in the License Agreement, by any means or
 * in any form be reproduced, distributed or lent.  Moreover, no part of the
 * program may be used, viewed, printed, disassembled or otherwise interfered
 * with in any form, except where allowed by the License Agreement, without the
 * express written consent of the copyright holder.
 *
 * THIS PROGRAM IS FURTHER PROVIDED "AS IS" WITH NO WARRANTY OF ANY KIND AND
 * THE COPYRIGHT HOLDER HEREBY DISCLAIMS ALL WARRANTIES, EXPRESS AND IMPLIED AS
 * TO THE PROGRAM.
 *
 * Copyright (c) 2002-2003 Sony Electronics Inc.
 * Some portions copyright (c) 1999 Palm Computing, Inc. or its subsidiaries.
 * All Rights Reserved.
 *
 * File: SecondForm.cpp
 *
 *****************************************************************************/


/** SecondForm -- a second Form for the SilkSample project;
  *               this Form supports resizing the virtual silkscreen area
  *                 and demonstrates how to handle custom drawing
  *                 approriately
  */

#include <PalmOS.h>
#include <SonyCLIE.h>

#include "SilkSample.h"
#include "SilkSampleRsc.h"



/* FUNCTION DEFINITIONS ------------------------------------------------ */

/***********************************************************************
 * FUNCTION:    PrvMoveSecondFormObjects
 * DESCRIPTION: moves the objects on the Second Form appropriately for
 *                the given change in window bounds
 * PARAMETERS:  IN/OUT frmP - pointer to the Second Form.
 *              dx          - the change in width of the window bounds
 *              dy          - the change in height of the window bounds
 * RETURNS:
 ***********************************************************************/
static void PrvMoveSecondFormObjects(FormType* frmP, Coord dx, Coord dy)
{
    if (gSilkState.refNum != sysInvalidRefNum && (dx != 0 || dy != 0))
    {
        // NOTE: be careful to avoid object migration!
        MoveFormObject(frmP, SecondDescriptionLabel, dx / 2, 0);

        MoveFormObject(frmP, SecondGoto1PushButton, dx, 0);
        MoveFormObject(frmP, SecondGoto2PushButton, dx, 0);
        MoveFormObject(frmP, SecondGoto3PushButton, dx, 0);
    }
}


/***********************************************************************
 * FUNCTION:    PrvSecondFormDoCustomDrawing
 * DESCRIPTION: performs custom drawing for the Second Form
 * PARAMETERS:  IN/OUT frmP - pointer to the Second Form.
 * RETURNS:
 ***********************************************************************/
static void PrvSecondFormDoCustomDrawing(FormType* frmP)
{
    WinHandle oldWinH = WinSetDrawWindow(FrmGetWindowHandle(frmP));
    WinPushDrawState();

    WinSetForeColor(UIColorGetTableEntryIndex(UIObjectForeground));
    WinSetBackColor(UIColorGetTableEntryIndex(UIObjectFill));

    RectangleType r;
    FrmGetFormBounds(frmP, &r);

    PointType bottomRight;
    bottomRight.x = r.topLeft.x + r.extent.x;
    bottomRight.y =  r.topLeft.y + r.extent.y;

    // draw an X across the entire Form
    WinDrawLine(r.topLeft.x, r.topLeft.y,
                bottomRight.x, bottomRight.y);
    WinDrawLine(bottomRight.x, r.topLeft.y,
                r.topLeft.x, bottomRight.y);

    WinPopDrawState();
    WinSetDrawWindow(oldWinH);
}


/***********************************************************************
 * FUNCTION:    PrvSecondFormInit
 * DESCRIPTION: This routine initializes the Second Form.
 * PARAMETERS:  IN/OUT frmP - pointer to the Second Form.
 * RETURNS:
 ***********************************************************************/
static void PrvSecondFormInit(FormType* frmP)
{
    if (gSilkState.refNum != sysInvalidRefNum)
    {
        if (gSilkState.resizeEnabled)
        {
            if (GetSilkPos(gSilkState.refNum) == silkResizeMax)
            {
                // This Form does not support hiding/showing the status
                // bar.  Therefore, restore the status bar if the
                // silkscreen state has been left in fullscreen mode.
                ResizeSilk(gSilkState.refNum, vskResizeMin);
            }
        }
        else // enable silkscreen resizing for this Form
        {
            // the silkscreen explicitly must be set to be resizable
            if (EnableSilkResize(gSilkState.refNum, vskResizeVertically | vskResizeHorizontally)
                == errNone)
            {
                gSilkState.resizeEnabled = true;
            }
        }

        // if resizeAmount.x != 0 || resizeAmount.y != 0, then we've
        // already resized the window and don't need to do it again
        if (gSilkState.resizeAmount.x == 0 && gSilkState.resizeAmount.y == 0)
        {
            ResizeWindowToDisplay(FrmGetWindowHandle(frmP), &gSilkState.resizeAmount.x, &gSilkState.resizeAmount.y);
        }

        PrvMoveSecondFormObjects(frmP, gSilkState.resizeAmount.x, gSilkState.resizeAmount.y);

        gSilkState.curResizableFormP = frmP;
        gSilkState.resizeAmount.x = gSilkState.resizeAmount.y = 0;
    }

    // initialize Form selection pushbuttons
    FrmSetControlGroupSelection(frmP, SecondGroupID, SecondGoto2PushButton);
}


/***********************************************************************
 * FUNCTION:    SecondFormHandleEvent
 * DESCRIPTION: This routine is the event handler for the Second Form
 * PARAMETERS:  IN eventP - a pointer to an EventType structure
 * RETURNS:     true if the event has been handled and should not be passed
 *              to a higher level handler.
 ***********************************************************************/
Boolean SecondFormHandleEvent(EventType* eventP)
{
    Boolean handled = false;

    FormType* frmP;

    switch (eventP->eType)
    {
        case menuEvent :
            handled = DoMenuCommand(eventP->data.menu.itemID);
            break;

        case frmOpenEvent :
            frmP = FrmGetActiveForm();

            // PrvSecondFormInit initializes the Form bounds and widget
            // positions
            PrvSecondFormInit(frmP);
            FrmDrawForm(frmP);
            PrvSecondFormDoCustomDrawing(frmP);

            handled = true;
            break;

        case frmUpdateEvent :
            // update the Form; reposition widgets and perform custom
            // drawing as necessary
            frmP = FrmGetActiveForm();

            if (gSilkState.resized)
            {
                // if a dialog was active when the silkscreen was resized,
                // the system may have cached the old clip bounds and may
                // not have restored them properly when the dialog was
                // dismissed
                WinResetClip();

                // when the silkscreen is minimized, the formerly occupied
                // area sometimes is not cleared with the Form's fill color
                //
                // this Form uses custom drawing, so we need to erase the
                // old stuff anyway
                IndexedColorType formFillColor,
                                 oldBackColor;
                formFillColor = UIColorGetTableEntryIndex(UIFormFill);
                oldBackColor = WinSetBackColor(formFillColor);
                WinEraseWindow();
                WinSetBackColor(oldBackColor);

                // (this Form does not have widgets that need to be
                // repositioned, but if it did, reposition them here)
                PrvMoveSecondFormObjects(frmP, gSilkState.resizeAmount.x, gSilkState.resizeAmount.y);

                gSilkState.resized = false;
                gSilkState.resizeAmount.x = gSilkState.resizeAmount.y = 0;
            }
            FrmDrawForm(frmP);

            // perform custom drawing
            PrvSecondFormDoCustomDrawing(frmP);
            handled = true;
            break;

        case winEnterEvent :
            // if we are returning from another window (such as a dialog
            // box), check to see if the Second Form needs updating as
            // a result of resizing the silkscreen area
            //
            // (we must check for winEnter events because modal Forms with
            // the save-behind bit enabled may not generate frmUpdate
            // events)
            if (   eventP->data.winEnter.enterWindow
                   == FrmGetWindowHandle(FrmGetActiveForm())
                && gSilkState.resized)
            {
                FrmUpdateForm(SecondForm, frmRedrawUpdateCode);
            }
            break;

        case ctlSelectEvent :
            handled = true;
            switch (eventP->data.ctlSelect.controlID)
            {
                case SecondGoto1PushButton :
                    FrmGotoForm(MainForm);
                    break;

                case SecondGoto2PushButton :
                    break;

                case SecondGoto3PushButton :
                    FrmGotoForm(ThirdForm);
                    break;

                default :
                    handled = false;
                    break;
            }
            break;

        default :
            break;
    }

    return handled;
}
