/******************************************************************************
 * This program is subject to copyright protection in accordance with the
 * applicable law and is also subject to the Developer Download Software
 * License Agreement, ("License Agreement") agreed to by the licensee prior to
 * download and receipt of this program.  It must only be used for the purpose
 * of developing software that is compatible with the CLI handheld computer.
 * It must not, except as permitted in the License Agreement, by any means or
 * in any form be reproduced, distributed or lent.  Moreover, no part of the
 * program may be used, viewed, printed, disassembled or otherwise interfered
 * with in any form, except where allowed by the License Agreement, without the
 * express written consent of the copyright holder.
 *
 * THIS PROGRAM IS FURTHER PROVIDED "AS IS" WITH NO WARRANTY OF ANY KIND AND
 * THE COPYRIGHT HOLDER HEREBY DISCLAIMS ALL WARRANTIES, EXPRESS AND IMPLIED AS
 * TO THE PROGRAM.
 *
 * Copyright (c) 2002-2003 Sony Electronics Inc.
 * Some portions copyright (c) 1999 Palm Computing, Inc. or its subsidiaries.
 * All Rights Reserved.
 *
 * File: SilkSample.cpp
 *
 *****************************************************************************/


/** SilkSample -- sample code to demonstrate basic usage of the Sony Silk
  *                 library and workarounds to common problems
  */

#include <PalmOS.h>
#include <PalmOSGlue.h>
#include <SonyCLIE.h>

#include "SilkSample.h"
#include "SilkSampleRsc.h"



/* GLOBALS ------------------------------------------------------------- */

UInt16 gHRRefNum = sysInvalidRefNum;

SilkStateType gSilkState;

// the original state of the silkscreen area when the application started
// static UInt16 gOriginalSilkPos = silkResizeNormal;



/* FUNCTION DEFINITIONS ------------------------------------------------ */

/***********************************************************************
 * FUNCTION:    GetSilkPos
 * DESCRIPTION: returns the current silkscreen position
 * PARAMETERS:  silkRefNum - reference number to the Silk Manager library
 * RETURNS:     silkResizeNormal, silkResizeToStatus, or silkResizeMax
 ***********************************************************************/
UInt16 GetSilkPos(UInt16 silkRefNum)
{
    UInt16 silkMgrVersion = VskGetAPIVersion(silkRefNum);
    if (silkMgrVersion < 2)
    {
        Coord w, h;

        // on Sony High-Resolution devices, WinGetDisplayExtent always returns
        // low-res (standard) values...
        WinGetDisplayExtent(&w, &h);

        // ... however, on high-density devices, we might need to scale the
        // dimensions
        UInt32 winMgrVersion;
        if (   FtrGet(sysFtrCreator, sysFtrNumWinVersion, &winMgrVersion) == errNone
            && winMgrVersion >= 4)
        {
            w = WinUnscaleCoord(w, false);
            h = WinUnscaleCoord(h, false);
        }

        switch (h)
        {
            default:
                ErrNonFatalDisplay("Unrecognized display extent");
                // fall through

            case stdHeight:
                return vskResizeMax;

            case stdHeight + stdSilkHeight:
                return vskResizeMin;

            case stdHeight + stdSilkHeight + stdStatusHeight:
                return vskResizeNone;
        }
    }
    else
    {
        UInt16 state;
        VskGetState(gSilkState.refNum, vskStateResize, &state);
        return state;
    }
}


/***********************************************************************
 * FUNCTION:    EnableSilkResize
 * DESCRIPTION: enables or disables silkscreen resizing
 * PARAMETERS:  silkRefNum - reference number to the Silk Manager library
 *              state      - the resize state to enable;
 *                           must be a vskResizeDisable or a combination of
 *                             the vskResizeVertically and
 *                             vskResizeHorizontally bit flags
 * RETURNS:     Err value
 ***********************************************************************/
Err EnableSilkResize(UInt16 silkRefNum, UInt16 state)
{
    Err error = errNone;
    UInt16 silkMgrVersion = VskGetAPIVersion(silkRefNum);
    if (silkMgrVersion < 2)
    {
        if (state & vskResizeVertically)
        {
            error = SilkLibEnableResize(silkRefNum);
        }
        else if (state == vskResizeDisable)
        {
            error = SilkLibDisableResize(silkRefNum);
        }
    }
    else
    {
        if (silkMgrVersion < 3)
        {
            // versions of the silk manager earlier than 3 do not recognize
            // the vskResizeHorizontally bit
            state &= vskResizeVertically;
        }
        error = VskSetState(silkRefNum, vskStateEnable, state);
    }
    return error;
}


/***********************************************************************
 * FUNCTION:    ResizeSilk
 * DESCRIPTION: resizes the virtual silkscreen to the specified size
 * PARAMETERS:  silkRefNum - reference number to the Silk Manager library
 *              size       - the desired silkscreen size;
 *                           must be vskResizeMax, vskResizeMin, or
 *                             vskResizeNone
 * RETURNS:     Err value
 ***********************************************************************/
Err ResizeSilk(UInt16 silkRefNum, UInt16 size)
{
    Err error = errNone;
    UInt16 silkMgrVersion = VskGetAPIVersion(silkRefNum);
    if (silkMgrVersion < 2)
    {
        error = SilkLibResizeDispWin(silkRefNum, size);
    }
    else
    {
        error = VskSetState(silkRefNum, vskStateResize, size);
    }
    return error;
}


/***********************************************************************
 * FUNCTION:    PrvHandleDisplayChange
 * DESCRIPTION: callback for sysNotifyDisplayChangeEvent notification;
 *              a sysNotifyDisplayChangeEvent notification is sent when
 *                the silkscreen area gets resized or when the screen
 *                mode/palette changes
 * PARAMETERS:  IN/OUT notifyParamsP - notification parameters
 * RETURNS:     errNone
 ***********************************************************************/
static Err PrvHandleDisplayChange(SysNotifyParamType* notifyParamsP)
{
    ErrFatalDisplayIf(notifyParamsP == NULL, "PrvHandleDisplayChange: null notification parameters");

    SysNotifyDisplayChangeDetailsType* detailsP
    = (SysNotifyDisplayChangeDetailsType*) notifyParamsP->notifyDetailsP;

    // when the silkscreen is resized, oldDepth == newDepth
    //
    // NOTE: this condition also will be true if the color palette has
    //       changed or if the screen resolution is changed
    if (detailsP->oldDepth == detailsP->newDepth)
    {
        SilkStateType* silkStateP = (SilkStateType*) notifyParamsP->userDataP;

        if (silkStateP->curResizableFormP != NULL)
        {
            WinHandle winH = FrmGetWindowHandle(silkStateP->curResizableFormP);

            // Keeping a variable around to store the change in display
            // height is admittedly a bit inelegant, but the window should
            // be resized ASAP to make sure the system redraws everything
            // properly when dialogs are dismissed.  The widget positioning
            // code should be handled by the respective Form's event
            // handler to make sure that all drawing becomes deferred.
            Coord dx, dy;
            ResizeWindowToDisplay(winH, &dx, &dy);
            silkStateP->resizeAmount.x += dx;
            silkStateP->resizeAmount.y += dy;

            // make a note that the silkscreen got resized and hasn't been
            // fully handled yet
            silkStateP->resized = true;

            // update the Form only if the current window is what we expect
            // it to be; otherwise, handling will be deferred until the
            // next winEnter or frmUpdate event
            //
            // (enqueuing a frmUpdate event while a modal dialog is active
            // may have no effect, because dialogs have their own event
            // loops and may intercept the frmUpdate event before the
            // intended Form can handle it)
            if (WinGetActiveWindow() == winH)
            {
                FrmUpdateForm(FrmGetFormId(silkStateP->curResizableFormP), frmRedrawUpdateCode);
            }
        }
    }

    notifyParamsP->handled = true;

    return errNone;
}


/***********************************************************************
 * FUNCTION:    ResizeWindowToDisplay
 * DESCRIPTION: resizes the bounds to the given window to match the bounds
 *                of the display window
 * PARAMETERS:  winH    - handle to the window
 *              OUT dxP - change in width;
 *                        pass NULL if you don't care about this value
 *              OUT dyP - change in height;
 *                        pass NULL if you don't care about this value
 * RETURNS:
 ***********************************************************************/
void ResizeWindowToDisplay(WinHandle winH, Coord* dxP, Coord* dyP)
{
    ErrFatalDisplayIf(winH == NULL, "ResizeWindowToDisplay: null Window");

    // the window bounds MUST be reset; otherwise, all drawing
    // will remain clipped to the previous dimensions
    RectangleType oldWinBounds,
                  winBounds;
    WinGetBounds(winH, &oldWinBounds);
    WinGetBounds(WinGetDisplayWindow(), &winBounds);

    // NOTE: make sure to specify explicitly which window's bounds are to
    //       be set; do not depend on it being the active window!
    WinSetBounds(winH, &winBounds);

    if (dxP != NULL) { *dxP = winBounds.extent.x - oldWinBounds.extent.x; }
    if (dyP != NULL) { *dyP = winBounds.extent.y - oldWinBounds.extent.y; }
}


/***********************************************************************
 * FUNCTION:    GetObjectPtr
 * DESCRIPTION: returns a pointer to an object in the given Form
 * PARAMETERS:  IN frmP  - pointer to the Form;
 *                         pass NULL to use the currently active Form
 *              objectID - ID number of the object in the Form
 * RETURNS:     void*
 ***********************************************************************/
void* GetObjectPtr(const FormType* frmP, UInt16 objectID)
{
    if (frmP == NULL) { frmP = FrmGetActiveForm(); }
    return FrmGetObjectPtr(frmP, FrmGetObjectIndex(frmP, objectID));
}


/***********************************************************************
 * FUNCTION:    GetGSIIndex
 * DESCRIPTION: retrieves the index of the Graffiti Shift Indicator on the
 *                given Form
 *              (FrmGetObjectIndex does not work on GSI objects)
 * PARAMETERS:  IN frmP - pointer to the Form
 * RETURNS:     the index of the Graffiti Shift Indicator, or
 *                frmInvalidObjectId if not found
 ***********************************************************************/
UInt16 GetGSIIndex(const FormType* frmP)
{
    ErrFatalDisplayIf(frmP == NULL, "GetGSIObjectIndex: null Form");

    UInt16 numObjects = FrmGetNumberOfObjects(frmP);
    for (UInt16 i = 0; i < numObjects; i++)
    {
        if (FrmGetObjectType(frmP, i) == frmGraffitiStateObj)
        {
            return i;
        }
    }

    return frmInvalidObjectId;
}


/***********************************************************************
 * FUNCTION:    MoveFormObject
 * DESCRIPTION: moves the specified Form object by a specified offset
 * PARAMETERS:  IN/OUT frmP - the Form containing the desired object
 *              id          - the ID of the object to move
 *              dx, dx      - the offset to move the Form object
 * RETURNS:     Err value
 ***********************************************************************/
void MoveFormObject(FormType* frmP, UInt16 id, Coord dx, Coord dy)
{
    UInt16 index = FrmGetObjectIndex(frmP, id);
    ErrFatalDisplayIf(index == frmInvalidObjectId, "object not found");

    RectangleType objBounds;
    FrmGetObjectBounds(frmP, index, &objBounds);
    FrmSetObjectPosition(frmP, index, objBounds.topLeft.x + dx, objBounds.topLeft.y + dy);
}


/***********************************************************************
 * FUNCTION:    PrvGetLibRefNum
 * DESCRIPTION: retrieves the reference number to the specified shared
 *                library
 * PARAMETERS:  IN nameP    - name of the library
 *              libType     - type of the library
 *              libCreator  - creator of the library
 *              OUT refNumP - stores the reference number to the specified
 *                              shared library
 *                            set to sysInvalidRefNum if there is an error
 * RETURNS:     Err value
 ***********************************************************************/
static Err PrvGetLibRefNum(const Char* nameP, UInt32 libType, UInt32 libCreator, UInt16* refNumP)
{
    ErrFatalDisplayIf(nameP == NULL, "PrvGetLibRefNum: null library name");
    ErrFatalDisplayIf(refNumP == NULL, "PrvGetLibRefNum: null reference number");

    Err error;
    if ((error = SysLibFind(nameP, refNumP)) == sysErrLibNotFound)
    {
        // library is not loaded yet
        error = SysLibLoad(libType, libCreator, refNumP);
    }

    if (error == errNone && (*refNumP == 0 || *refNumP == sysInvalidRefNum))
    {
        error = sysErrLibNotFound;
    }
    else if (error != errNone)
    {
        *refNumP = sysInvalidRefNum;
    }

    return error;
}


/***********************************************************************
 * FUNCTION:    PrvSetScreenModeToHR
 * DESCRIPTION: retrieves the reference number to the Sony High-Resolution
 *                library and sets the screen mode to high-resolution
 * PARAMETERS:  OUT hrRefNumP - stores the reference number to the Sony HR
 *                                library
 *                              set to sysInvalidRefNum if there is an
 *                                error
 * RETURNS:     Err value
 ***********************************************************************/
static Err PrvSetScreenModeToHR(UInt16* hrRefNumP)
{
    ErrFatalDisplayIf(hrRefNumP == NULL, "PrvSetScreenModeToHR: null reference number");

    Err error = errNone;

    // load the Sony high-resolution library
    error = PrvGetLibRefNum(sonySysLibNameHR, sonySysFileTHRLib, sonySysFileCHRLib,
                            hrRefNumP);

    // set the screen mode to high-resolution
    if (error == errNone)
    {
        if ((error = HROpen(*hrRefNumP)) == errNone)
        {
            UInt32 w = hrWidth,
                   h = hrHeight;
            if ((error = HRWinScreenMode(*hrRefNumP, winScreenModeSet, &w, &h, NULL, NULL))
                != errNone)
            {
                HRClose(*hrRefNumP);
            }
        }
    }

    if (error != errNone)
    {
        *hrRefNumP = sysInvalidRefNum;
    }

    return error;
}


/***********************************************************************
 * FUNCTION:    PrvLoadSilkLib
 * DESCRIPTION: retrieves the reference number to the Sony Silk library
 *                and registers for the sysNotifyDisplayChangeEvent
 *                notification;
 *              does NOT enable silkscreen resizing;
 *              do NOT call this function before setting the screen mode
 * PARAMETERS:  OUT silkRefNumP  - stores the reference number to the Sony
 *                                   Silk library
 *                                 set to sysInvalidRefNum if there is an
 *                                   error
 *              IN/OUT userDataP - user data for the
 *                                   sysNotifyDisplayChangeEvent
 *                                   notification
 * RETURNS:     Err value
 ***********************************************************************/
static Err PrvLoadSilkLib(UInt16* silkRefNumP, void* userDataP)
{
    ErrFatalDisplayIf(silkRefNumP == NULL, "PrvLoadSilkLib: null reference number");

    Err error = errNone;

    error = PrvGetLibRefNum(sonySysLibNameSilk, sonySysFileTSilkLib, sonySysFileCSilkLib,
                             silkRefNumP);

    if (error == errNone)
    {
        // the silkscreen library is available and usable
        if ((error = VskOpen(*silkRefNumP)) == errNone) // same as SilkLibOpen
        {
            // register for sysNotifyDisplayChangeEvent notifications;
            // a sysNotifyDisplayChangeEvent notification is sent when the
            // user resizes the silkscreen area
            // NOTE: these notifications must be registered AFTER changing
            //       the screen resolution/bit-depth
            UInt16 cardNo;
            LocalID dbID;
            if (   (error = SysCurAppDatabase(&cardNo, &dbID)) != errNone
                || (error = SysNotifyRegister(cardNo, dbID, sysNotifyDisplayChangeEvent, NULL,
                                              sysNotifyNormalPriority, userDataP))
                   != errNone)
            {
                ErrNonFatalDisplayIf(error != errNone,
                                     "failed to register for sysNotifyDisplayChangeEvent");
                VskClose(*silkRefNumP); // same as SilkLibClose
            }
        }
    }

    if (error != errNone)
    {
        *silkRefNumP = sysInvalidRefNum;
    }

    return error;
}


/***********************************************************************
 * FUNCTION:    PrvRomVersionCompatible
 * DESCRIPTION: This routine checks that a ROM version is meet your
 *                minimum requirement.
 * PARAMETERS:  requiredVersion - minimum rom version required
 *                                (see sysFtrNumROMVersion in SystemMgr.h
 *                                  for format)
 *              launchFlags     - flags that indicate if the application
 *                                  UI is initialized
 * RETURNS:     Err value; errNone if ROM is compatible
 ***********************************************************************/
static Err PrvRomVersionCompatible(UInt32 requiredVersion, UInt16 launchFlags)
{
    UInt32 romVersion;

    // ensure that the device is using the required version of the ROM or
    // later
    FtrGet(sysFtrCreator, sysFtrNumROMVersion, &romVersion);
    if (romVersion < requiredVersion)
    {
        if ((launchFlags & (sysAppLaunchFlagNewGlobals | sysAppLaunchFlagUIApp))
            == (sysAppLaunchFlagNewGlobals | sysAppLaunchFlagUIApp))
        {
            FrmAlert(RomIncompatibleAlert);

            // Palm OS 1.0 will continuously relaunch this app unless we
            // switch to a safe one.
            if (romVersion < sysMakeROMVersion(2, 0, 0, sysROMStageRelease, 0))
            {
                AppLaunchWithCommand(sysFileCDefaultApp,
                                     sysAppLaunchCmdNormalLaunch, NULL);
            }
        }

        return sysErrRomIncompatible;
    }

    return errNone;
}


/***********************************************************************
 * FUNCTION:    DoMenuCommand
 * DESCRIPTION: This routine performs the menu command specified.
 *              This sample application uses one menu bar for all Forms.
 * PARAMETERS:  command - menu item ID
 * RETURNS:     true if the menu command has been handled
 ***********************************************************************/
Boolean DoMenuCommand(UInt16 command)
{
    Boolean handled = false;

    FormType* frmP;

    MenuEraseStatus(NULL);

    switch (command)
    {
        case OptionsAbout:
            // display the About dialog
            frmP = FrmInitForm(AboutForm);
            FrmDoDialog(frmP);
            FrmDeleteForm(frmP);
            handled = true;
            break;
    }

    return handled;
}


/***********************************************************************
 * FUNCTION:    PrvAppHandleEvent
 * DESCRIPTION: This routine loads form resources and sets the form event
 *                handlers.
 * PARAMETERS:  IN event - a pointer to the event to handle
 * RETURNS:     true if the event has been handled and should not be passed
 *              to a higher level handler.
 ***********************************************************************/
static Boolean PrvAppHandleEvent(EventType* eventP)
{
    Boolean handled = false;

    UInt16 formID;
    FormType* frmP;

    if (eventP->eType == frmLoadEvent)
    {
        // load the form resource
        formID = eventP->data.frmLoad.formID;
        frmP = FrmInitForm(formID);
        FrmSetActiveForm(frmP);

        // Set the event handler for the form.  The handler of the
        // currently active form is called by FrmHandleEvent each time it
        // receives an event.
        handled = true;
        switch (formID)
        {
            case MainForm:
                FrmSetEventHandler(frmP, MainFormHandleEvent);
                break;

            case SecondForm:
                FrmSetEventHandler(frmP, SecondFormHandleEvent);
                break;

            case ThirdForm:
                FrmSetEventHandler(frmP, ThirdFormHandleEvent);
                break;

            default:
                // ErrFatalDisplay("Invalid Form Load Event");
                handled = false;
                break;
        }
    }

    return handled;
}


/***********************************************************************
 * FUNCTION:    PrvAppEventLoop
 * DESCRIPTION: This routine is the main event loop for the application.
 * PARAMETERS:
 * RETURNED:
 ***********************************************************************/
static void PrvAppEventLoop(void)
{
    Err error;
    EventType event;

    do
    {
        EvtGetEvent(&event, evtWaitForever);

        if (!SysHandleEvent(&event))
        {
            if (!MenuHandleEvent(0, &event, &error))
            {
                if (!PrvAppHandleEvent(&event))
                {
                    FrmDispatchEvent(&event);
                }
            }
        }
    } while (event.eType != appStopEvent);
}


/***********************************************************************
 * FUNCTION:     PrvAppStart
 * DESCRIPTION:  Gets the application's preferences and initializes
 *                 necessary data structures.
 * PARAMETERS:
 * RETURNS:      Err value
 ***********************************************************************/
static Err PrvAppStart(void)
{
    // initialize global structures
    gSilkState.refNum = sysInvalidRefNum;
    gSilkState.resizeEnabled = false;
    gSilkState.resized = false;
    gSilkState.resizeAmount.x = gSilkState.resizeAmount.y = 0;
    gSilkState.curResizableFormP = NULL;

    // try to set the screen mode to high-resolution if possible
    // (the virtual silkscreen is usable in low-resolution mode, but
    // the status bar will not be available, and users will not be able to
    // resize the silkscreen area manually)
    (void) PrvSetScreenModeToHR(&gHRRefNum);

    // load the Sony Silk library
    if (PrvLoadSilkLib(&gSilkState.refNum, &gSilkState) != errNone)
    {
        FrmCustomAlert(ErrOKAlert, "Silk library not available", "", "");

        // allow the application to continue running if you want to support
        // devices that do not have the library
    }
    // gOriginalSilkPos = GetSilkPos(gSilkState.refNum);

    return errNone;
}


/***********************************************************************
 * FUNCTION:    PrvAppStop
 * DESCRIPTION: Saves the current state of the application and frees used
 *                system resources.
 * PARAMETERS:
 * RETURNS:
 ***********************************************************************/
static void PrvAppStop(void)
{
    // close all open forms
    FrmCloseAllForms();

    // unregister sysNotifyDisplayChangeEvent notification
    UInt16 cardNo;
    LocalID dbID;
    SysCurAppDatabase(&cardNo, &dbID);
    SysNotifyUnregister(cardNo, dbID, sysNotifyDisplayChangeEvent, sysNotifyNormalPriority);

    if (gSilkState.refNum != sysInvalidRefNum)
    {
        if (   gSilkState.resizeEnabled
            || EnableSilkResize(gSilkState.refNum, vskResizeVertically | vskResizeHorizontally)
               == errNone)
        {
            if (VskGetAPIVersion(gSilkState.refNum) < 2)
            {
                // old devices didn't restore the silkscreen automatically
                // for applications that don't support silkscreen resizing;
                // therefore, we need to restore the silkscreen explicitly
                (void) ResizeSilk(gSilkState.refNum, vskResizeNone);
            }
            else
            {
                // make sure we don't leave the user stuck in fullscreen mode
                if (GetSilkPos(gSilkState.refNum) == silkResizeMax)
                {
                    (void) ResizeSilk(gSilkState.refNum, vskResizeMin);
                }
            }

            // disable silkscreen resizing before we quit
            (void) EnableSilkResize(gSilkState.refNum, vskResizeDisable);
        }

        // close the silkscreen library
        (void) VskClose(gSilkState.refNum); // same as SilkLibClose
    }

    // restore the screen mode to the system defaults
    if (gHRRefNum != sysInvalidRefNum)
    {
        HRWinScreenMode(gHRRefNum, winScreenModeSetToDefaults, NULL, NULL, NULL, NULL);
        HRClose(gHRRefNum);
    }
    else
    {
        WinScreenMode(winScreenModeSetToDefaults, NULL, NULL, NULL, NULL);
    }
}


/***********************************************************************
 * FUNCTION:    PilotMain
 * DESCRIPTION: This is the main entry point for the application.
 * PARAMETERS:  cmd         - value specifying the launch code
 *              IN cmdPBP   - pointer to a structure specific to the launch
 *                              code
 *              launchFlags - extra information about the launch
 * RETURNS:     Result of launch
 ***********************************************************************/
UInt32 PilotMain(UInt16 cmd, void* cmdPBP, UInt16 launchFlags)
{
    Err error = PrvRomVersionCompatible(MIN_OS_VERSION, launchFlags);
    if (error != errNone) { return error; }

    switch (cmd)
    {
        case sysAppLaunchCmdNormalLaunch:
            error = PrvAppStart();
            if (error == errNone)
            {
                FrmGotoForm(MainForm);
                PrvAppEventLoop();
            }
            PrvAppStop();
            break;

        case sysAppLaunchCmdNotify:
        {
            ErrFatalDisplayIf(cmdPBP == NULL,
                              "SilkSample: PilotMain: null notification parameters");

            SysNotifyParamType* notifyParamsP = (SysNotifyParamType*) cmdPBP;
            switch (notifyParamsP->notifyType)
            {
                case sysNotifyDisplayChangeEvent:
                {
                    Boolean appIsActive = (Boolean) (launchFlags & sysAppLaunchFlagSubCall);
                    if (appIsActive)
                    {
                        PrvHandleDisplayChange(notifyParamsP);
                    }
                    break;
                }
            }
            break;
        }
        default:
            break;
    }

    return error;
}
