/***********************************************************************
 Sample Code Disclaimer

 Copyright  2001 Palm, Inc. or its subsidiaries.  All
 rights reserved.

 You may incorporate this sample code (the "Code") into your applications
 for Palm OS(R) platform products and may use the Code to develop
 such applications without restriction.  The Code is provided to you on
 an "AS IS" basis and the responsibility for its operation is 100% yours.
 PALM, INC. AND ITS SUBSIDIARIES (COLLECTIVELY, "PALM") DISCLAIM
 ALL WARRANTIES, TERMS AND CONDITIONS WITH RESPECT TO THE CODE, EXPRESS,
 IMPLIED, STATUTORY OR OTHERWISE, INCLUDING WARRANTIES, TERMS OR
 CONDITIONS OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE,
 NONINFRINGEMENT AND SATISFACTORY QUALITY.  You are not permitted to
 redistribute the Code on a stand-alone basis and you may only
 redistribute the Code in object code form as incorporated into your
 applications.  TO THE FULL EXTENT ALLOWED BY LAW, PALM ALSO EXCLUDES ANY
 LIABILITY, WHETHER BASED IN CONTRACT OR TORT (INCLUDING NEGLIGENCE), FOR
 INCIDENTAL, CONSEQUENTIAL, INDIRECT, SPECIAL OR PUNITIVE DAMAGES OF ANY
 KIND, OR FOR LOSS OF REVENUE OR PROFITS, LOSS OF BUSINESS, LOSS OF
 INFORMATION OR DATA, OR OTHER FINANCIAL LOSS ARISING OUT OF OR IN
 CONNECTION WITH THE USE OR PERFORMANCE OF THE CODE.  The Code is subject
 to Restricted Rights for U.S. government users and export regulations.
 
 SAMPLE NAME:		HostFS

 FILE:				HostFS.c

 DATE:				6/25/2001

 DESCRIPTION:		Source Code

 COMPILER:			Metrowerks CodeWarrior

 *****************************************************************************
 
 Description: POSE Host file system library implementation.

 *****************************************************************************/

// Include Palm headers
#include <PalmTypes.h>
#include <ErrorMgr.h>
#include <FeatureMgr.h>
#include <Preferences.h>
#include <SystemMgr.h>
#include <SoundMgr.h>
#include <SysUtils.h>
#include <StringMgr.h>
#include <NotifyMgr.h>
#include <Form.h>

#include <HostControl.h>

// Our library public definitions
#include "VFSMgr.h"
#include "FSLib.h"
#include "SlotDrvrLib.h"



/* Note:  if the path to the POSE emulator directory is longer than the PATHPREFIX (or even close to the
   same length) there's a serious chance that no file could ever be found.  We could just increase this constant
   if that happens. */
#define PATHPREFIXLEN 256


/* Note:  The comments in this file documenting the various file operations are distinctly out of date, and
   maintaining them is likely a poor idea;  there is a repository for them already in another file, and
   trying to keep both copies up to date is a significant and unnecessary waste of effort.
   Trust pretty much any source of documentation in preference to this file for function docs. */

/* Known Design Bugs:
	There is no really reasonable way to calculate available and used space on the "volume" HostFS mounts, so
	no serious attempt is made and instead dummy values are returned.

	Many standard FAT FS attributes map questionably to the HostControl API operations.
	For example, setting the system bit on a Macintosh file is unlikely to have a useful effect on the file.
	
	HostFS doesn't directly enforce locking, or prevent things like deleting a file that is open, but rather
	relies on the mechanisms of the underlying OS and filesystem to do so.  The results vary according
	to platform and filesystem.
	
	HostFS doesn't support, and likely will never support, formatting of the host disk.  In addition to the
	lack of HostControl operations for this purpose, this would be a nontrivial security risk.
	
	HostFS trusts the POSE HostControl API to handle all timezone information.  When/if better timezone
	support exists, this could be fixed.
	
	Can't set the volume label -- this is a meaningless operation on Mac or Unix, and HostControl
	doesn't allow it on Windows.
*/

/* Known non-design bugs:
	Keeping multiple copies of the same file open and setting dates on more than one of the FileRefs can lose
	previous changes.
	
	Filenames containing characters which are directory-separators on other platforms (like colons in a
	Unix filename or slashes in a Mac filename) are handled improperly.
*/

/* Minor bugs:

	The volume is mounted at the root directory for Unix/Windows systems and at the current directory
	for Macintosh.  This is an artifact of the path conventions of these platforms, and of POSE not
	supporting a single, uniform path string format.  Now that POSE has a UI for setting the path to the root
	of the volume, this problem still exists but is irrelevant for regular users.
*/

// ************************************** Private Structures ********************************************

#define kHostFSLibCreator	'hstf'

/* Flags for HostFSFILE attributes field */
#define HOSTFLAG_ISDIR     0x1
#define HOSTFLAG_ISLINK    0x2
#define HOSTFLAG_READONLY  0x4
#define HOSTFLAG_HIDDEN    0x8
#define HOSTFLAG_SYSTEM    0x10

typedef struct _HostFSFILE_struct {
	HostFILE*                  hostFile;
	HostDIRType*               hostDir;
	struct _HostFSFILE_struct* next;
	struct _HostFSFILE_struct* prev;
	struct _HostFSMountedVolume_struct* volume;
	Char*                      filename;
	UInt16                     attributes;
	Int32                      size;
	HostUTimeType              times;
} HostFSFILE;


// Linked list of mounted volumes
typedef struct _HostFSMountedVolume_struct {
	UInt16		                        volumeRef;		           // 0, or volRefNum if the volume is mounted
	Char                                volprefix[PATHPREFIXLEN];  // Prefix for mounted volume
	
	HostFSFILE*                         openFile;                  // First of linked list of open files on volume

	struct _HostFSMountedVolume_struct* next, *prev;
} HostFSMountedVolumeType;

typedef HostFSMountedVolumeType* HostFSMountedVolumePtr;


// 

/********************************************************************
 * LIBRARY GLOBALS:
 *
 * IMPORTANT:
 * ==========
 * Libraries are *not* allowed to have global or static variables.  Instead,
 * they allocate a memory chunk to hold their persistent data, and save
 * a handle to it in the library's system library table entry.  Example
 *	functions below demonstrate how the library "globals" chunk is set up, saved,
 * and accessed.
 *
 ********************************************************************/


// This value may need to be altered for a copy of POSE with more slots than listed here -- POSE currently supports 4 or 8, depending
#define HOSTFS_MAX_SLOTS   12

typedef struct {
	Int32		           openCount;			        // library open count
	Char                   separator;                   // Directory separator, runtime-determined, platform specific
	Char                   pathprefix[PATHPREFIXLEN];   // Path prefix for OS ("C:", perhaps), used to generate volume prefix
	UInt8		           rwBuffer[slotSectorSize];	// buffer we use to read & write blocks from the slot driver
	char                   pathBuffer[1024];
	char                   renamePathBuffer[1024];
	UInt16                 poseCardVolRef[HOSTFS_MAX_SLOTS + 1];  // VFS SlotNum of each POSE SlotNum
	int                    numSlots;					// Current number of slots that POSE thinks exist -- potentially dynamic

	HostFSMountedVolumePtr volume;
} HostFSGlobalsType;

typedef HostFSGlobalsType *HostFSGlobalsPtr;



// ************************************** Functions ********************************************
Err PrvCopyBool(void* bufBaseP, UInt32 offset, void *srcP, UInt32 numBytes, Boolean DataStoreBased);
Err PrvFSErrEquiv(HostErr hostError);
Err PrvFSPathConvert(const Char* oldPath, Char* newPath, Char separator, Char* pathPrefix, Char* volprefix);
Char* PrvFindFilename(Char *path);
Err PrvFindMountedVolume(HostFSGlobalsPtr gP, UInt16 volumeRef, HostFSMountedVolumePtr* mountedVolumeP);
Err PrvCloseAllFiles(HostFSFILE* openFile);
void PrvAdjustNumSlots(HostFSGlobalsPtr gP);
static void PrvMountPOSEVolumes(HostFSGlobalsPtr gP);
static void PrvUnmountPOSEVolumes(HostFSGlobalsPtr gP);
static Err PrvResetNotifyProc(SysNotifyParamType* notifyParamsP);
static Err PrvMountNotifyProc(SysNotifyParamType* notifyParamsP);
static void PrvVolumeVerify(HostFSGlobalsPtr gP);

#define DO_PATH_CONVERSION(str)         PrvFSPathConvert(str, gP->pathBuffer, gP->separator, gP->pathprefix, mV->volprefix); \
										str = gP->pathBuffer;


/********************************************************************
 * Standard library open, close, sleep and wake functions
 ********************************************************************/

/************************************************************
 *
 *  FUNCTION:	FSLibOpen
 *
 *
 *  DESCRIPTION:		Opens the FS library, creates and initializes the globals.
 *						This function must be called before any other FS Library functions,
 *						with the exception of FSLibAPIVersion.
 *
 *						If FSLibOpen fails, do not call any other FS library API functions.
 *						If FSLibOpen succeeds, call FSLibClose when you are done using
 *						the library to enable it to release critical system resources.
 *						Clients (usually VFS Lib) should call FSLibOpen() before using the
 *						filesystem driver.
 *
 *  LIBRARY DEVELOPER NOTES:
 *						The library's "open" and "close" functions should *not* take an excessive
 *						amount of time to complete.  If the processing time for either of these
 *						is lengthy, consider creating additional library API function(s) to handle
 *						the time-consuming chores.
 *
 *
 *  PARAMETERS:	fsLibRefNum	-- FS library reference number returned by SysLibLoad()
 *								or SysLibFind().
 *
 *  CALLED BY:		anyone who wants to use this library
 *
 *  RETURNS:		errNone					-- no error
 *					memErrNotEnoughSpace	-- not enough memory to initialize
 *
 *************************************************************/ 
Err FSLibOpen(UInt16 fsLibRefNum)
{
	Err                    error = errNone;
	Err                    finalErr;
	HostFSGlobalsPtr       gP;
	int                    ctr;
	
	gP = SysLibTblEntry(fsLibRefNum)->globalsP;
	
	if (!gP) {
		gP = MemPtrNew(sizeof(HostFSGlobalsType));
		if (!gP) {
			return sysErrNoFreeRAM;
		}
		
		// MemPtrNew sets the 'owner' of a chunk to whatever the current application is.
		// When that app exits, all chunks owned by it are disposed of.
		// Call MemPtrSetOwner() to set the owner to the unused/system one so that the
		// chunk will not be disposed of when the current app exits.
		MemPtrSetOwner(gP, 0);
		MemSet(gP, sizeof(HostFSGlobalsType), 0);
		SysLibTblEntry(fsLibRefNum)->globalsP = gP;
		
		gP->volume = NULL;
		
		switch(HostGetHostPlatform())
		{
			case hostPlatformWindows:
				gP->separator = '\\';
				gP->pathprefix[0] = '\0';
				break;
			case hostPlatformMacintosh:
				gP->separator = ':';
				gP->pathprefix[0] = '\0';
				break;
			case hostPlatformUnix:
				gP->separator = '/';
				gP->pathprefix[0] = '\0';
				break;
			case hostPlatformPalmOS:
				/* Don't care, I guess - won't work on PalmOS */
				gP->separator = ' ';
				gP->pathprefix[0] = '\0';
				ErrNonFatalDisplay("Running POSE HostFS library on a Palm.  That makes no sense!");
				break;
			default:
				ErrFatalDisplay("Unrecognized POSE platform in FSLibOpen()!");
				return -1;
		}
		for(ctr=0; ctr<HOSTFS_MAX_SLOTS + 1; ctr++) {
			gP->poseCardVolRef[ctr] = 0;
		}
		/* Default directories... */
		finalErr = errNone;
		error = VFSRegisterDefaultDirectory("image/jpeg", expMediaType_PoserHost, "/jpegs/");
		if(error != errNone) finalErr = error;
		error = VFSRegisterDefaultDirectory(".jpeg", expMediaType_PoserHost, "/jpegs/");
		if(error != errNone) finalErr = error;
		error = VFSRegisterDefaultDirectory(".jpg", expMediaType_PoserHost, "/jpegs/");
		if(error != errNone) finalErr = error;
		error = VFSRegisterDefaultDirectory("image/tiff", expMediaType_PoserHost, "/pictures/");
		if(error != errNone) finalErr = error;
		error = VFSRegisterDefaultDirectory(".tiff", expMediaType_PoserHost, "/pictures/");
		if(error != errNone) finalErr = error;
		error = VFSRegisterDefaultDirectory(".tif", expMediaType_PoserHost, "/pictures/");
		if(error != errNone) finalErr = error;
		// Register for reset-finished notification, so we can mount POSE volumes:
		error = SysNotifyRegister(0, sysNotifyNoDatabaseID, sysNotifyResetFinishedEvent, 
									PrvResetNotifyProc, sysNotifyNormalPriority, (void*)gP);
		if(error != errNone) finalErr = error;
		ErrFatalDisplayIf(finalErr != errNone, "HostFS: FSLibOpen can't register for notify");
		error = finalErr;
	}
	gP->openCount++;					// increment open count

	return error;
}


/************************************************************
 *
 *  FUNCTION:	FSLibClose
 *
 *  DESCRIPTION:	Closes the FS libary, frees client context and globals.
 *					Clients (usually VFS Lib) should call FSLibClose() after using the
 *					filesystem driver.
 *
 *						***IMPORTANT***
 *						May be called only if FSLibOpen succeeded.
 *
 *						If other applications still have the library open, decrements
 *						the reference count and returns expErrStillOpen.
 *
 *
 *  LIBRARY DEVELOPER NOTES:
 *						The library's "open" and "close" functions should *not* take an excessive
 *						amount of time to complete.  If the processing time for either of these
 *						is lengthy, consider creating additional library API function(s) to handle
 *						the time-consuming chores.
 *							
 *
 *  PARAMETERS:	fsLibRefNum		-- FS library reference number returned by SysLibLoad()
 *									or SysLibFind().
 *  CALLED BY:	Whoever wants to close the FS library (usually ExpansionLib or file system drivers)
 *				should call this after using the driver
 *
 *  RETURNS:	errNone			-- no error
 *				expErrStillOpen	-- library is still open by others (no error)
 *				expErrNotOpen	-- library is not open
 *
 *************************************************************/
Err FSLibClose(UInt16 fsLibRefNum)
{
	Err error = errNone;
	HostFSGlobalsPtr gP;
	
	gP = SysLibTblEntry(fsLibRefNum)->globalsP;
	if (!gP || !gP->openCount)
		return expErrNotOpen;

	if (--gP->openCount)
		return expErrStillOpen;

	// If open count is zero, remove globals
	if (gP->openCount == 0) {
		PrvUnmountPOSEVolumes(gP);
		MemPtrFree(gP);
	}
	
	// What would I do if one of these returned an error?  Re-registering should break nothing, so for the
	// moment I'll ignore it.
	error = VFSUnregisterDefaultDirectory("image/jpeg", expMediaType_PoserHost);
	error = VFSUnregisterDefaultDirectory(".jpeg", expMediaType_PoserHost);
	error = VFSUnregisterDefaultDirectory(".jpg", expMediaType_PoserHost);
	error = VFSUnregisterDefaultDirectory("image/tiff", expMediaType_PoserHost);
	error = VFSUnregisterDefaultDirectory(".tiff", expMediaType_PoserHost);
	error = VFSUnregisterDefaultDirectory(".tif", expMediaType_PoserHost);
	
	return error;
}


/************************************************************
 *
 *  FUNCTION:	FSLibSleep
 *
 *  DESCRIPTION:	Handles system sleep notification.
 *
 *						***IMPORTANT***
 *						This notification function is called from a system interrupt.
 *						It is only allowed to use system services which are interrupt-
 *						safe.  Presently, this is limited to EvtEnqueueKey, SysDisableInts,
 *						SysRestoreStatus.  Because it is called from an interrupt,
 *						it must *not* take a long time to complete to preserve system
 *						integrity.  The intention is to allow system-level libraries
 *						to disable hardware components to conserve power while the system
 *						is asleep.
 *
 *  PARAMETERS:	fsLibRefNum		-- FS library reference number
 *
 *  CALLED BY:	System when going to sleep.
 *
 *  RETURNS:	errNone			-- no error
 *				expErrNotOpen	-- FS driver library has not been opened
 *
 *************************************************************/
Err FSLibSleep(UInt16)
{
	return errNone;
}


/************************************************************
 *
 *  FUNCTION:	FSLibWake
 *
 *  DESCRIPTION:	Handles system wake notification
 *
 *						***IMPORTANT***
 *						This notification function is called from a system interrupt.
 *						It is only allowed to use system services which are interrupt-
 *						safe.  Presently, this is limited to EvtEnqueueKey, SysDisableInts,
 *						SysRestoreStatus.  Because it is called from an interrupt,
 *						it must *not* take a long time to complete to preserve system
 *						integrity.  The intention is to allow system-level libraries
 *						to enable hardware components which were disabled when the system
 *						went to sleep.
 *
 *  PARAMETERS:	fsLibRefNum		-- FS library reference number
 *
 *  CALLED BY:	System when waking up.
 *
 *  RETURNS:	errNone			-- no error
 *				expErrNotOpen	-- FS driver library has not been opened
 *
 *************************************************************/
Err FSLibWake(UInt16)
{
	return errNone;
}


/********************************************************************
 * Custom library API functions
 ********************************************************************/

/************************************************************
 *
 *  FUNCTION:	FSLibAPIVersion
 *
 *  DESCRIPTION:	Return version of the library API
 *					library need not be open to call
 *
 *  PARAMETERS:	fsLibRefNum	-- FS library reference number
 *
 *  RETURNS:	32-bit API version
 *
 *************************************************************/
UInt32 FSLibAPIVersion(UInt16 fsLibRefNum)
{
#pragma unused(fsLibRefNum)
	return fsLibAPIVersion;
}


/************************************************************
 *
 *  FUNCTION:	FSCustomControl
 *
 *  DESCRIPTION:	Handles a custom call for a particular volume.
 *
 *  LIBRARY DEVELOPER NOTES:
 *		The driver identifies the call and its API by a registered creator code and a selector.
 *		This allows filesystem developers to extend the API by defining selectors for their 
 *		creator code. It also allows filesystem developers to suppport selectors (and custom calls) 
 *		defined by other filesystem developers.
 *
 *  PARAMETERS:	fsLibRefNum				-- FS library reference number
 * 				apiCreator				-- registered creator code
 * 				apiSelector				-- custom operation to perform
 * 				valueP					-- buffer containing data specific to the operation
 * 				valueLenP				-- size of the valueP buffer on entry,
 *											size of data written to valueP on exit
 *
 *  RETURNS:	errNone					-- no error
 *				expErrNotOpen			-- FS driver library has not been opened
 *				expErrUnsupportedOperation	-- for all unsupported or undefined opcodes
 *												and/or creators.
 *				sysErrParamErr			-- valueP buffer is too small
 *				Sets the size of the returned valueP buffer in valueLenP
 *
 *************************************************************/
#define PRIVATE_STACKSIZE 2048
Err FSCustomControl(UInt16 fsLibRefNum, UInt32 apiCreator, UInt16 apiSelector, 
					void *valueP, UInt16 *valueLenP)
{
// This function doesn't use valueLenP for Release builds
#pragma unused(valueLenP)
	HostFSGlobalsPtr gP;
	Err error = errNone;
	int poseSlotNum;
	SysNotifyParamType notify;
	
	gP = SysLibTblEntry(fsLibRefNum)->globalsP;
	if (!gP)
		return expErrNotOpen;

	if(apiCreator != 'pose') return expErrUnsupportedOperation;

	poseSlotNum = (int)(valueP);
	if(apiSelector == 1) {
		// Mount volume
		
		PrvAdjustNumSlots(gP);

		if((poseSlotNum < 0) || (poseSlotNum >= HOSTFS_MAX_SLOTS)) return expErrUnsupportedOperation;
		ErrNonFatalDisplayIf(valueLenP, "ValueLenP != NULL in VFSCustomControl!");
		
		//error = PrvVFSVolumeMount(0, 0, (VFSAnyMountParamPtr)&mountParam);
		notify.notifyType = sysNotifyPOSEMountEvent;
		notify.broadcaster = kHostFSLibCreator;
		notify.notifyDetailsP = (void*)((UInt32)valueP | 0x10000L);
		error = SysNotifyBroadcastDeferred(&notify, 0);
		return error;
	} else if(apiSelector == 0) {
		PrvAdjustNumSlots(gP);

		if((poseSlotNum < 0) || (poseSlotNum >= HOSTFS_MAX_SLOTS)) return expErrUnsupportedOperation;
		ErrNonFatalDisplayIf(valueLenP, "ValueLenP != NULL in VFSCustomControl!");

		//error = PrvVFSVolumeUnmount(gP->poseCardVolRef[poseSlotNum]);

		notify.notifyType = sysNotifyPOSEMountEvent;
		notify.broadcaster = kHostFSLibCreator;
		notify.notifyDetailsP = (void*)valueP;
		error = SysNotifyBroadcastDeferred(&notify, 0);

		return error;
	} else {
		return expErrUnsupportedOperation;
	}

	return error;
}

/************************************************************
 *
 *  FUNCTION:	FSFilesystemType
 *
 *  DESCRIPTION:	Return the type of filesystem that this library implements.
 *
 *  PARAMETERS:	fsLibRefNum				-- FS library reference number
 *				filesystemTypeP			-- On return set to the type of filesystem
 *											Common types are defined in VFSMgr.h
 *
 *  RETURNS:	errNone					-- no error
 *				expErrNotOpen			-- FS driver library has not been opened
 *				Fills in filesystemTypeP with the type of filesystem that this library implements.
 *
 *************************************************************/
Err FSFilesystemType(UInt16 /*fsLibRefNum*/, UInt32 *filesystemTypeP)
{
	if (filesystemTypeP != NULL)
		*filesystemTypeP = 'Pose';
	return errNone;
}


/********************************************************************
 * File Stream APIs:
 ********************************************************************/
 
/************************************************************
 *
 *  FUNCTION:	FSFileCreate
 *
 *  DESCRIPTION:	Create a file.  Does not open the file.
 *					All parts of the path, except the last part must already exist.
 *
 *  LIBRARY DEVELOPER NOTES:
 *		It is the responsibility of the filesystem library to ensure that all filenames
 *		are 'mangled' into a format that is compatible with the native format of
 *		the filesystem.  (i.e. 8.3 for a FAT filesystem without long filename support)
 *		See comment above on "Filename conventions"
 *		Note that this function does not open the file.
 *
 *  PARAMETERS:	fsLibRefNum		-- FS library reference number
 *				volRefNum		-- Volume reference number passed to FSVolumeMount
 *				pathNameP		-- Full path of the file to be created
 *
 *  RETURNS:	errNone					-- no error
 *				expErrNotOpen			-- FS driver library has not been opened
 *				vfsErrFileAlreadyExists	-- A file of this name exists already in this location
 *				vfsErrVolumeBadRef		-- the volume has not been mounted with FSVolumeMount
 *				vfsErrVolumeFull		-- not enough space left on volume
 *				vfsErrBadName			-- pathNameP is invalid
 *
 *************************************************************/
Err FSFileCreate(UInt16 fsLibRefNum, UInt16 volRefNum, const Char *pathNameP)
{
	HostFSGlobalsPtr       gP;
	Err                    err = errNone;
	HostFILE*              hfp;
	HostFSMountedVolumePtr mV;

	gP = SysLibTblEntry(fsLibRefNum)->globalsP;
	if (!gP)
		return expErrNotOpen;

	PrvFindMountedVolume(gP, volRefNum, &mV);

	if(gP->volume == NULL) 
		return vfsErrVolumeBadRef;

	if(volRefNum != mV->volumeRef) 
		return vfsErrVolumeBadRef;

	if(pathNameP[0] != '/')
		return vfsErrBadName;

	DO_PATH_CONVERSION(pathNameP);
	
	// Check to see if file already exists
	hfp = HostFOpen(pathNameP, "r");
	if(hfp)
	{
		HostFClose(hfp);
		return vfsErrFileAlreadyExists;
	}

	// Write new file onto media:
	hfp = HostFOpen(pathNameP, "w");
	if(!hfp)
	{
		err = HostErrNo();
		if(err == hostErrFileNotFound) return vfsErrDirectoryNotFound;
		if(err == hostErrInvalidParameter) return vfsErrBadName;
		return PrvFSErrEquiv(err);
	}
	err = HostFClose(hfp);
	if(err != errNone) return PrvFSErrEquiv(err);

	return errNone;
}


/************************************************************
 *
 *  FUNCTION:	FSFileOpen
 *
 *  DESCRIPTION:	Open a file or directory.
 *					Note the the FileRef obtained for a directory can not be used
 *					for all functions.  For example, it is not permitted (or logical)
 *					to read directly from an opened directory.
 *					openMode is ignored when opening a directory.
 *
 *  PARAMETERS:	fsLibRefNum				-- FS library reference number
 *				volRefNum				-- Volume reference number passed to FSVolumeMount
 *				pathNameP				-- Full path of the file to be opened
 *											This must all be valid -- non-null, non-empty
 *				openMode				-- fsMode to use when opening the file.  Ignored for directories
 *				fileRefP				-- Pointer to a reference to the opened file
 *											Filled in on return.
 *
 *  RETURNS:	errNone					-- no error
 *				expErrNotOpen			-- FS driver library has not been opened
 *				vfsErrFilePermissionDenied	-- The file or directory can not be opened with the requested
 *												openMode or has already been opened with vfsModeExclusive
 *				vfsErrFileNotFound		-- the file could not be found
 *				vfsErrVolumeBadRef		-- the volume has not been mounted with FSVolumeMount
 *				expErrCardReadOnly		-- card is read only, but openMode includes vfsModeWrite
 *				vfsErrBadName			-- pathNameP is invalid
 *				Fills in the passed in fileRefP with the file reference number
 *
 *************************************************************/
Err FSFileOpen(UInt16 fsLibRefNum, UInt16 volRefNum, const Char *pathNameP,
	UInt16 openMode, FileRef *fileRefP)
{
	HostFSGlobalsPtr       gP;
	HostFILE*              hfp;
	char                   hostOpenMode[6];
	char                   hostOpenIndex;
	HostFSMountedVolumePtr mV;
	HostFSFILE*            hF;
	char                   isDir = false;
	HostDIRType*           dirP;
	HostStatType           stat;
	HostErr                hostErr;
	long                   hostAttrs;

	gP = SysLibTblEntry(fsLibRefNum)->globalsP;
	if (!gP)
		return expErrNotOpen;

	PrvFindMountedVolume(gP, volRefNum, &mV);
	if(mV == NULL)
		return vfsErrVolumeBadRef;
	
	if(volRefNum != mV->volumeRef) 
		return vfsErrVolumeBadRef;
		
	if(pathNameP[0] != '/')
		return vfsErrBadName;
	
	DO_PATH_CONVERSION(pathNameP);
	
	/* Create hostOpenMode for HostFOpen call */
	hostOpenIndex = 0;
	if(openMode & vfsModeRead)
	{
		hostOpenMode[0] = 'r';
		hostOpenIndex++;
		if(openMode & vfsModeWrite)
		{
			hostOpenMode[1] = '+';
			hostOpenIndex++;
		}
	}
	else
	{
		/* This truncates the file */
		hostOpenMode[0] = 'w';
		hostOpenIndex++;
	}
	
	/* Open as binary - add a 'b' to the string */
	hostOpenMode[hostOpenIndex] = 'b';
	hostOpenIndex++;
	
	hostOpenMode[hostOpenIndex] = '\0';  /* NULL-terminate the string */
	
	hfp = HostFOpen(pathNameP, hostOpenMode);
	if(!hfp)
	{
		hostErr = HostErrNo();
		if(hostErr == errNone)
		{
			ErrNonFatalDisplay("NULL HostFILE* returned but no error in errno.  Makes no sense!");
		}
		
		// Didn't work as a file, let's see if it's a directory
		dirP = HostOpenDir(pathNameP);
		if(dirP)
		{
			isDir = true;
		}
		else
		{
			return PrvFSErrEquiv(hostErr);
		}
	}

	hF = MemPtrNew(sizeof(HostFSFILE));
	if(!hF)
	{
		HostFClose(hfp);
		return sysErrNoFreeRAM;
	}
	MemPtrSetOwner(hF, 0);

	hF->hostFile = NULL;
	hF->hostDir = NULL;
	if(isDir) hF->hostDir = dirP;
	else hF->hostFile = hfp;
	hF->attributes = isDir ? HOSTFLAG_ISDIR : 0;
	hF->filename = MemPtrNew(StrLen(pathNameP) + 1);
	StrCopy(hF->filename, pathNameP);
	MemPtrSetOwner(hF->filename, 0);
	
	// Now stat the file to grab attributes...
	hostErr = HostStat(pathNameP, &stat);
	if(hostErr == errNone)
	{
		hF->size = stat.st_size_;
		hF->times.crtime_ = stat.st_ctime_;
		hF->times.actime_ = stat.st_atime_;
		hF->times.modtime_ = stat.st_mtime_;
		if(stat.st_mode_ & 0120000)
			hF->attributes |= HOSTFLAG_ISLINK;
	}
	
	hostErr = HostGetFileAttr(pathNameP, &hostAttrs);
	if(hostErr == errNone) {
		if(hostAttrs & hostFileAttrReadOnly) {
			hF->attributes |= HOSTFLAG_READONLY;
		}
		if(hostAttrs & hostFileAttrHidden) {
			hF->attributes |= HOSTFLAG_HIDDEN;
		}
		if(hostAttrs & hostFileAttrSystem) {
			hF->attributes |= HOSTFLAG_SYSTEM;
		}
	}
	
	hF->next = mV->openFile;
	hF->prev = NULL;
	if(hF->next) hF->next->prev = hF;
	hF->volume = mV;
	mV->openFile = hF;
	
	*fileRefP = (FileRef)hF;

	return errNone;
}


/************************************************************
 *
 *  FUNCTION:	FSFileClose
 *
 *  DESCRIPTION:	Close a file or directory which has been opened with FSFileOpen
 *
 *  PARAMETERS:	fsLibRefNum		-- FS library reference number
 *				fileRef			-- File reference number returned from FSFileOpen
 *
 *  RETURNS:	errNone					-- no error
 *				expErrNotOpen			-- FS driver library has not been opened
 *				vfsErrFileBadRef		-- the fileref is invalid
 *
 *************************************************************/
Err FSFileClose(UInt16 fsLibRefNum, FileRef fileRef)
{
	HostFSGlobalsPtr       gP;
	HostErr                err;
	HostFSFILE*            hff;
	
	gP = SysLibTblEntry(fsLibRefNum)->globalsP;
	if (!gP)
		return expErrNotOpen;

	hff = (HostFSFILE*)fileRef;
	MemPtrFree(hff->filename);
	if(hff->attributes & HOSTFLAG_ISDIR)
	{
		err = HostCloseDir(hff->hostDir);
		if(err == hostErrInvalidParameter) return vfsErrFileBadRef;
		if(err != errNone) return PrvFSErrEquiv(err);
	}
	else
	{
		err = HostFClose(hff->hostFile);
		if(err == hostErrInvalidParameter) return vfsErrFileBadRef;
		if(err != errNone) return PrvFSErrEquiv(err);
	}
	if(hff->prev == NULL) {
		hff->volume->openFile = hff->next;
	} else {
		hff->prev->next = hff->next;
	}
	if(hff->next) {
		hff->next->prev = hff->prev;
	}
	MemPtrFree(hff);
	return errNone;
}


/************************************************************
 *
 *  FUNCTION:	FSFileRead
 *
 *  DESCRIPTION:	Read data from an open file into a buffer or a data storage heap-based
 *					chunk (record or resource).
 *					This function is not permitted for FileRefs to open directories.
 *
 *  LIBRARY DEVELOPER NOTES:
 *					This function must check the boolean dataStoreBased to determine if bufBaseP is a pointer
 *					to a storage heap based chunk.  If it is, this function must use DmWrite to write to the
 *					buffer.
 *
 *  PARAMETERS:	fsLibRefNum			-- FS library reference number
 *				fileRef				-- File reference number returned from FSFileOpen
 *				numBytes			-- The number of bytes to read
 *				bufBaseP			-- ptr to beginning of destination buffer for reading data (must be
 *										beginning of record or resource for data storage heap based destination)
 *				offset				-- offset from base ptr to the destination area
 *				dataStoreBased		-- if true, the bufBaseP points to a buffer in the storage heap
 *										if false the bufBaseP buffer is in the dynamic heap
 *				numBytesReadP		-- Set to the number of bytes actually read on return if non-NULL
 *										This does not need to be initialized on input.
 *
 *  RETURNS:	errNone				-- no error
 *				expErrNotOpen		-- FS driver library has not been opened
 *				vfsErrFileBadRef	-- the fileref is invalid
 *              vfsErrIsADirectory  -- the fileref points to a directory
 *				vfsErrFileEOF		-- file pointer is at end of file
 *				vfsErrFilePermissionDenied	-- read permission is not enabled for this file
 *				Returns the number of bytes actually read in numBytesReadP if it is not NULL
 *
 *************************************************************/
Err FSFileRead(UInt16 fsLibRefNum, FileRef fileRef, UInt32 numBytes, void *bufBaseP, 
						UInt32 offset, Boolean DataStoreBased, UInt32 *numBytesReadPtr)
{
	HostFSGlobalsPtr       gP;
	Err                    err = errNone;
	long                   length;

	if(numBytesReadPtr) *numBytesReadPtr = 0;

	gP = SysLibTblEntry(fsLibRefNum)->globalsP;
	if (!gP)
		return expErrNotOpen;
		
	if (((HostFSFILE*)fileRef)->attributes & HOSTFLAG_ISDIR)
		return vfsErrIsADirectory;

	while(numBytes > 0)
	{
		long bytesToRead;

		// read a block:
		bytesToRead = (numBytes > slotSectorSize) ? slotSectorSize : numBytes;
		length = HostFRead(gP->rwBuffer, 1, bytesToRead, ((HostFSFILE*)fileRef)->hostFile);
		if(length != bytesToRead)
		{
			if(numBytesReadPtr && length >= 0) *numBytesReadPtr += length;
			err = HostFError(((HostFSFILE*)fileRef)->hostFile);
			if(err == errNone)
			{
				err = PrvCopyBool(bufBaseP, offset, gP->rwBuffer, length, DataStoreBased);

				if(err == errNone)
				{
					if(HostFEOF(((HostFSFILE*)fileRef)->hostFile))
					{
						err = vfsErrFileEOF;
					}
					else
					{
						ErrDisplay("Unknown error in FSFileRead");
						err = vfsErrFileGeneric;  // What in the world is this?
					}
				}
			}
			return err;
		}

		// copy data into user buffer
		err = PrvCopyBool(bufBaseP, offset, gP->rwBuffer, length, DataStoreBased);
		if(err != errNone) return err;
		offset += length;

		// update loop variables:
		if(numBytesReadPtr) *numBytesReadPtr += length;
		numBytes -= length;
	}
	
	return err;
}


/************************************************************
 *
 *  FUNCTION:	FSFileWrite
 *
 *  DESCRIPTION:	Write data to an open file
 *					This function is not permitted for FileRefs to open directories.
 *
 *  PARAMETERS:	fsLibRefNum			-- FS library reference number
 *				fileRef				-- File reference number returned from FSFileOpen
 *				numBytes			-- The number of bytes to write
 *				dataP				-- ptr to data to write
 *				numBytesWrittenP	-- Set to the number of bytes actually written on return if non-NULL
 *										This does not need to be initialized on input.
 *
 *  RETURNS:	errNone				-- no error
 *				expErrNotOpen		-- FS driver library has not been opened
 *				vfsErrFileBadRef	-- the fileref is invalid
 *              vfsErrIsADirectory  -- the fileref points to a directory
 *				vfsErrFilePermissionDenied	-- write permission is not enabled for this file
 *				Returns the number of bytes actually written in numBytesWrittenP if it is not NULL
 *
 *************************************************************/
Err FSFileWrite(UInt16 fsLibRefNum, FileRef fileRef, 
				UInt32 numBytes, const void *dataP, UInt32 *numBytesWrittenP)
{
	HostFSGlobalsPtr       gP;
	long                   length;

	if(numBytesWrittenP) *numBytesWrittenP = 0;

	gP = SysLibTblEntry(fsLibRefNum)->globalsP;
	if (!gP)
		return expErrNotOpen;

	if(((HostFSFILE*)fileRef)->attributes & HOSTFLAG_ISDIR)
		return vfsErrIsADirectory;

	// write onto media:
	length = HostFWrite(dataP, 1,  numBytes, ((HostFSFILE*)fileRef)->hostFile);
	if(length != numBytes)
	{
		if(numBytesWrittenP && length >= 0) *numBytesWrittenP += length;
		return PrvFSErrEquiv(HostFError(((HostFSFILE*)fileRef)->hostFile));
	}

	if(numBytesWrittenP) *numBytesWrittenP = numBytes;

	return errNone;
}


/************************************************************
 *
 *  FUNCTION:	FSFileDelete
 *
 *  DESCRIPTION:	Delete a closed file or directory.
 *
 *  PARAMETERS:	fsLibRefNum				-- FS library reference number
 *				volRefNum				-- Volume reference number passed to FSVolumeMount
 *				pathNameP				-- Full path of the file or directory to be deleted
 *
 *  RETURNS:	errNone					-- no error
 *				expErrNotOpen			-- FS driver library has not been opened
 *				vfsErrFileStillOpen		-- File is still open
 *				vfsErrFileNotFound		-- the file could not be found 
 *				vfsErrVolumeBadRef		-- the volume has not been mounted with FSVolumeMount
 *				vfsErrDirNotEmpty		-- (Only when the FileRef points to a directory)
 *											can't delete a non-empty directory
 *				vfsErrFilePermissionDenied	-- write permission is not enabled for this file
 *				vfsErrBadName			-- pathNameP is invalid
 *
 *************************************************************/
Err FSFileDelete(UInt16 fsLibRefNum, UInt16 volRefNum, const Char *pathNameP)
{
	HostFSGlobalsPtr       gP;
	HostErr                err = errNone;
	HostFSMountedVolumePtr mV;

	gP = SysLibTblEntry(fsLibRefNum)->globalsP;
	if (!gP)
		return expErrNotOpen;
		
	PrvFindMountedVolume(gP, volRefNum, &mV);
	if(mV == NULL)
		return vfsErrVolumeBadRef;
	
	if(volRefNum != mV->volumeRef) 
		return vfsErrVolumeBadRef;
	
	if(pathNameP[0] != '/')
		return vfsErrBadName;
	
	DO_PATH_CONVERSION(pathNameP);
	err = HostRemove(pathNameP);
	if(err != errNone)
	{
		HostErr err2;
		err2 = HostRmDir(pathNameP);
		if(err2 == errNone) return errNone;
		return PrvFSErrEquiv(err);
	}

	return errNone;
}


/************************************************************
 *
 *  FUNCTION:	FSFileRename
 *
 *  DESCRIPTION:	Rename a closed file or directory.
 *
 *  PARAMETERS:	fsLibRefNum				-- FS library reference number
 *				volRefNum				-- Volume reference number passed to FSVolumeMount
 *				pathNameP				-- Full path of the file or directory to be renamed
 *				newNameP				-- new file name only (not a full path)
 *
 *  RETURNS:	errNone					-- no error
 *				expErrNotOpen			-- FS driver library has not been opened
 *				vfsErrFileStillOpen		-- File is still open
 *				vfsErrFileNotFound		-- the file could not be found
 *				vfsErrFileAlreadyExists	-- A file of this name exists already in this location
 *				vfsErrVolumeBadRef		-- the volume has not been mounted with FSVolumeMount
 *				vfsErrFilePermissionDenied	-- write permission is not enabled for this file
 *				vfsErrBadName			-- pathNameP or newNameP is invalid
 *
 *************************************************************/
Err FSFileRename(UInt16 fsLibRefNum, UInt16 volRefNum, const Char *pathNameP, const Char *newNameP)
{
	HostFSGlobalsPtr       gP;
	Err                    err = errNone;
	HostFSMountedVolumePtr mV;
	Char*                  index;
	
	gP = SysLibTblEntry(fsLibRefNum)->globalsP;
	if (!gP)
		return expErrNotOpen;
		
	PrvFindMountedVolume(gP, volRefNum, &mV);
	if(mV == NULL)
		return vfsErrVolumeBadRef;
	
	if(volRefNum != mV->volumeRef) 
		return vfsErrVolumeBadRef;
	
	if(pathNameP[0] != '/')
		return vfsErrBadName;
	
	DO_PATH_CONVERSION(pathNameP);
	StrCopy(gP->renamePathBuffer, pathNameP);
	index = PrvFindFilename(gP->renamePathBuffer);
	StrCopy(index, newNameP);

	newNameP = gP->renamePathBuffer;
	err = HostRename(pathNameP, newNameP);
	if(err != errNone) return PrvFSErrEquiv(err);
	
	return errNone;
}


/************************************************************
 *
 *  FUNCTION:	FSFileSeek
 *
 *  DESCRIPTION:	Set position within an open file.  If the resulting
 *					position would be beyond the end of the file, sets the
 *					position to the end of file.
 *					This function is not permitted for FileRefs to open directories.
 *
 *  PARAMETERS:	fsLibRefNum			-- FS library reference number
 *				fileRef				-- File reference number returned from FSFileOpen
 *				origin				-- origin to use when calculating new position from the offset
 *				offset				-- offset from the origin to set the new position in the file
 *
 *  RETURNS:	errNone				-- no error
 *				vfsErrFileEOF		-- file pointer is at end of file (not an error)
 *				expErrNotOpen		-- FS driver library has not been opened
 *				vfsErrFileBadRef	-- the fileref is invalid
 *              vfsErrIsADirectory  -- the fileref points to a directory
 *				sysErrParamErr		-- origin is invalid
 *
 *************************************************************/
Err FSFileSeek(UInt16 fsLibRefNum, FileRef fileRef, FileOrigin origin, Int32 offset)
{
	HostFSGlobalsPtr       gP;
	HostErr                err;
	int                    newOrigin;

	gP = SysLibTblEntry(fsLibRefNum)->globalsP;
	if (!gP)
		return expErrNotOpen;

	if (((HostFSFILE*)fileRef)->attributes & HOSTFLAG_ISDIR)
		return vfsErrIsADirectory;
	
	switch(origin)
	{
		case vfsOriginBeginning:
			newOrigin = 0;
			break;
		case vfsOriginCurrent:
			newOrigin = 1;
			break;
		case vfsOriginEnd:
			newOrigin = 2;
			break;
		default:
			return sysErrParamErr;
	}
	err = HostFSeek(((HostFSFILE*)fileRef)->hostFile, offset, newOrigin);
	if(err != errNone) return PrvFSErrEquiv(err);

	return errNone;
}


/************************************************************
 *
 *  FUNCTION:	FSFileEOF
 *
 *  DESCRIPTION:	Get end-of-file status of an open file.
 *					This function is not permitted for FileRefs to open directories.
 *
 *  PARAMETERS:	fsLibRefNum			-- FS library reference number
 *				fileRef				-- File reference number returned from FSFileOpen
 *
 *  RETURNS:	errNone				-- file pointer is NOT at end of file
 *				vfsErrFileEOF		-- file pointer is at end of file
 *				expErrNotOpen		-- FS driver library has not been opened
 *				vfsErrFileBadRef	-- the fileref is invalid
 *              vfsErrIsADirectory  -- the fileref points to a directory
 *
 *************************************************************/
Err FSFileEOF(UInt16 fsLibRefNum, FileRef fileRef)
{
	HostFSGlobalsPtr       gP;
	Err                    error = errNone;

	gP = SysLibTblEntry(fsLibRefNum)->globalsP;
	if (!gP)
		return expErrNotOpen;
	
	if (((HostFSFILE*)fileRef)->attributes & HOSTFLAG_ISDIR)
		return vfsErrIsADirectory;
	
	if(HostFEOF(((HostFSFILE*)fileRef)->hostFile)) return vfsErrFileEOF;
	return errNone;
}


/************************************************************
 *
 *  FUNCTION:	FSFileTell
 *
 *  DESCRIPTION:	Get current position of the file pointer within an open file.
 *					This function is not permitted for FileRefs to open directories.
 *
 *  PARAMETERS:	fsLibRefNum			-- FS library reference number
 *				fileRef				-- File reference number returned from FSFileOpen
 *				filePosP			-- Receives the current file position
 *
 *  RETURNS:	errNone				-- no error
 *				expErrNotOpen		-- FS driver library has not been opened
 *				vfsErrFileBadRef	-- the fileref is invalid
 *              vfsErrIsADirectory  -- the fileref points to a directory
 *				Sets filePosP with the current file position.
 *
 *************************************************************/
Err FSFileTell(UInt16 fsLibRefNum, FileRef fileRef, UInt32 *filePosP)
{
	HostFSGlobalsPtr       gP;
	long                   pos;

	gP = SysLibTblEntry(fsLibRefNum)->globalsP;
	if (!gP)
		return expErrNotOpen;
		
	if(((HostFSFILE*)fileRef)->attributes & HOSTFLAG_ISDIR)
		return vfsErrIsADirectory;

	pos = HostFTell(((HostFSFILE*)fileRef)->hostFile);
	if(pos < 0)
	{
		return PrvFSErrEquiv(HostErrNo());
	}
	*filePosP = pos;

	return errNone;
}


/************************************************************
 *
 *  FUNCTION:	FSFileResize
 *
 *  DESCRIPTION:	Change size of an open file.
 *					If the resizing of the file would make the current
 *					file pointer point beyond EOF, the file pointer will
 *					be set to point to EOF.
 *					This function is not permitted for FileRefs to open directories.
 *
 *  PARAMETERS:	fsLibRefNum			-- FS library reference number
 *				fileRef				-- File reference number returned from FSFileOpen
 *				newSize				-- The desired new size of the file.
 *										This can be bigger or smaller then the current file size.
 *
 *  RETURNS:	errNone				-- no error
 *				expErrNotOpen		-- FS driver library has not been opened
 *				vfsErrFileBadRef	-- the fileref is invalid
 *              vfsErrIsADirectory  -- the fileref points to a directory
 *
 *************************************************************/
#define ZERO_CHUNK_SIZE 32
Err FSFileResize(UInt16 fsLibRefNum, FileRef fileRef, UInt32 newSize)
{
	HostFSGlobalsPtr       gP;
	UInt32                 curSize;
	HostErr                err;
	long                   filePos;

	gP = SysLibTblEntry(fsLibRefNum)->globalsP;
	if (!gP)
		return expErrNotOpen;

	if(((HostFSFILE*)fileRef)->attributes & HOSTFLAG_ISDIR)
		return vfsErrIsADirectory;

	err = FSFileSize(fsLibRefNum, fileRef, &curSize);
	if(err != errNone) return PrvFSErrEquiv(err);

	filePos = HostFTell(((HostFSFILE*)fileRef)->hostFile);
	if(filePos < 0) { return PrvFSErrEquiv(HostErrNo()); }
	if(newSize < curSize)
	{
		err = HostTruncate(((HostFSFILE*)fileRef)->filename, newSize);
		if(err != errNone) return PrvFSErrEquiv(err);
	} else {
		char buf[ZERO_CHUNK_SIZE];
		int i, len;
		
		for(i=0; i<ZERO_CHUNK_SIZE; i++) buf[i] = 0;
		HostFSeek(((HostFSFILE*)fileRef)->hostFile, 0, 2);  /* Seek to end of file */
		for(i=0; i<((newSize - curSize) / ZERO_CHUNK_SIZE); i++)
		{
			len = HostFWrite(buf, ZERO_CHUNK_SIZE, 1, ((HostFSFILE*)fileRef)->hostFile);
			if(len <= 0) return PrvFSErrEquiv(HostErrNo()); 
		}
		len = HostFWrite(buf, ((newSize - curSize) % ZERO_CHUNK_SIZE), 1, ((HostFSFILE*)fileRef)->hostFile);
		if(len <= 0) return PrvFSErrEquiv(HostErrNo());
	}
	HostFSeek(((HostFSFILE*)fileRef)->hostFile, filePos, 0);
	
	return errNone;
}


/************************************************************
 *
 *  FUNCTION:	FSFileGetAttributes
 *
 *  DESCRIPTION:	Obtain the attributes of an open file or directory.
 *
 *  PARAMETERS:	fsLibRefNum			-- FS library reference number
 *				fileRef				-- File reference number returned from FSFileOpen
 *				attributesP			-- File or directory attributes, filled in on return
 *										These are defined in VFSMgr.h
 *
 *  RETURNS:	errNone				-- no error
 *				expErrNotOpen		-- FS driver library has not been opened
 *				vfsErrFileBadRef	-- the fileref is invalid
 *				Sets attributesP to the open file or directory attributes
 *
 *************************************************************/
Err FSFileGetAttributes(UInt16 fsLibRefNum, FileRef fileRef, UInt32 *attributesP)
{
	HostFSGlobalsPtr       gP;
	Err                    error = errNone;

	gP = SysLibTblEntry(fsLibRefNum)->globalsP;
	if (!gP)
		return expErrNotOpen;
	
	*attributesP = 0; // no special attributes
	
	if(((HostFSFILE*)fileRef)->attributes & HOSTFLAG_ISDIR)
		*attributesP |= vfsFileAttrDirectory;
	if(((HostFSFILE*)fileRef)->attributes & HOSTFLAG_READONLY)
		*attributesP |= vfsFileAttrReadOnly;
	if(((HostFSFILE*)fileRef)->attributes & HOSTFLAG_ISLINK)
		*attributesP |= vfsFileAttrLink;
	if(((HostFSFILE*)fileRef)->attributes & HOSTFLAG_HIDDEN)
		*attributesP |= vfsFileAttrHidden;
	if(((HostFSFILE*)fileRef)->attributes & HOSTFLAG_SYSTEM)
		*attributesP |= vfsFileAttrSystem;
	
	return errNone;
}


/************************************************************
 *
 *  FUNCTION:	FSFileSetAttributes
 *
 *  DESCRIPTION:	Change the attributes of an open file or directory.
 *					Cannot use this function to set the fsAttribDirectory or fsAttribVolumeLabel
 *					attributes.  Use FSDirCreate and FSVolumeLabelSet.
 *
 *  PARAMETERS:	fsLibRefNum			-- FS library reference number
 *				fileRef				-- File reference number returned from FSFileOpen
 *				attributes			-- The file attributes to set to the file
 *
 *  RETURNS:	errNone				-- no error
 *				expErrNotOpen		-- FS driver library has not been opened
 *				vfsErrFileBadRef	-- the fileref is invalid
 *				sysErrParamErr		-- attributes included fsAttribDirectory or fsAttribVolumeLabel
 *
 *************************************************************/
Err FSFileSetAttributes(UInt16 fsLibRefNum, FileRef fileRef, UInt32 attributes)
{
	HostFSGlobalsPtr gP;
	HostFSFILE*      hF;
	long             hostAttr;
	HostErr          hE;

	if (attributes & (vfsFileAttrDirectory | vfsFileAttrVolumeLabel | vfsFileAttrLink))
		return sysErrParamErr;
	
	gP = SysLibTblEntry(fsLibRefNum)->globalsP;
	if (!gP)
		return expErrNotOpen;
	
	hF = (HostFSFILE*)fileRef;
	hostAttr = 0;

	if(attributes & HOSTFLAG_READONLY) hostAttr |= hostFileAttrReadOnly;
	if(attributes & HOSTFLAG_SYSTEM) hostAttr |= hostFileAttrSystem;
	if(attributes & HOSTFLAG_HIDDEN) hostAttr |= hostFileAttrHidden;

	hE = HostSetFileAttr(hF->filename, hostAttr);
	if(hE == errNone) {
		hF->attributes &= ~(HOSTFLAG_READONLY | HOSTFLAG_SYSTEM | HOSTFLAG_HIDDEN);
		if(hF->attributes & HOSTFLAG_READONLY) hF->attributes |= HOSTFLAG_READONLY;
		if(hF->attributes & HOSTFLAG_HIDDEN) hF->attributes |= HOSTFLAG_HIDDEN;
		if(hF->attributes & HOSTFLAG_SYSTEM) hF->attributes |= HOSTFLAG_SYSTEM;
	
		return errNone;
	}
	return PrvFSErrEquiv(hE);
}


/************************************************************
 *
 *  FUNCTION:	FSFileGetDate
 *
 *  DESCRIPTION:	Obtain the dates of an open file or directory.
 *
 *  PARAMETERS:	fsLibRefNum			-- FS library reference number
 *				fileRef				-- File reference number returned from FSFileOpen
 *				whichDate			-- Specifies which date to return.  These are defined in VFSMgr.h
 *				dateP				-- Receives the requested date
 *
 *  RETURNS:	errNone				-- no error
 *				expErrNotOpen		-- FS driver library has not been opened
 *				vfsErrFileBadRef	-- the fileref is invalid
 *				sysErrParamErr		-- whichDate is not a defined constant
 *				Sets dateP with the date specified with whichDate 
 *
 *************************************************************/
 
// In the 66 years from Jan 1904 to Jan 1970, there are 17 leap days.
// FILEDATEMOD is the number of seconds from Jan 1, 1904 to Jan 1, 1970, and is the amount added to an
// ANSI standard file time to get to the one we want to return.
#define FILEDATEMOD ((((1970-1904)*365) + 17) * (60*60*24))
Err FSFileGetDate(UInt16 fsLibRefNum, FileRef fileRef, 
					UInt16 dateSelector, UInt32 *outDate)
{
	HostFSGlobalsPtr gP;
	UInt32 date;
//	Err error = errNone;
	
	gP = SysLibTblEntry(fsLibRefNum)->globalsP;
	if (!gP)
		return expErrNotOpen;
		
	if(dateSelector == vfsFileDateCreated)
	{
		date = ((HostFSFILE*)fileRef)->times.crtime_ + FILEDATEMOD;
		*outDate = date;
		return errNone;
	}
	else if(dateSelector == vfsFileDateModified)
	{
		date = ((HostFSFILE*)fileRef)->times.modtime_ + FILEDATEMOD;
		*outDate = date;
		return errNone;
	}
	else if(dateSelector == vfsFileDateAccessed)
	{
		date = ((HostFSFILE*)fileRef)->times.actime_ + FILEDATEMOD;
		*outDate = date;
		return errNone;
	}	

	return expErrUnsupportedOperation;
}


/************************************************************
 *
 *  FUNCTION:	FSFileSetDate
 *
 *  DESCRIPTION:	Change the dates of an open file or directory.
 *
 *  PARAMETERS:	fsLibRefNum			-- FS library reference number
 *				fileRef				-- File reference number returned from FSFileOpen
 *				whichDate			-- Specifies which date to set.  These are defined in VFSMgr.h
 *				date				-- Contains the date to set
 *
 *  RETURNS:	errNone				-- no error
 *				expErrNotOpen		-- FS driver library has not been opened
 *				vfsErrFileBadRef	-- the fileref is invalid
 *				sysErrParamErr		-- whichDate is not a defined constant
 *
 *************************************************************/
Err FSFileSetDate(UInt16 fsLibRefNum, FileRef fileRef,
					UInt16 dateSelector, UInt32 inDate)
{
	HostFSGlobalsPtr gP;
	UInt16  status;
	HostErr hostErr = errNone;
	
	gP = SysLibTblEntry(fsLibRefNum)->globalsP;
	if (!gP)
		return expErrNotOpen;

	switch(dateSelector)
	{
	case vfsFileDateCreated:
		((HostFSFILE*)fileRef)->times.crtime_ = inDate - FILEDATEMOD;
		break;
	case vfsFileDateAccessed:
		((HostFSFILE*)fileRef)->times.actime_ = inDate - FILEDATEMOD;
		break;
	case vfsFileDateModified:
		((HostFSFILE*)fileRef)->times.modtime_ = inDate - FILEDATEMOD;
		break;
	default:
		return expErrUnsupportedOperation;
	}
	status = HostUTime(((HostFSFILE*)fileRef)->filename, &(((HostFSFILE*)fileRef)->times));
	if(status < 0)
		hostErr = HostErrNo();

	return PrvFSErrEquiv(hostErr);
}


/************************************************************
 *
 *  FUNCTION:	FSFileSize
 *
 *  DESCRIPTION:	Obtain the size of an open file.
 *					This function is not permitted for FileRefs to open directories.
 *
 *  PARAMETERS:	fsLibRefNum			-- FS library reference number
 *				fileRef				-- File reference number returned from FSFileOpen
 *				fileSizeP			-- Receives the size of the open file
 *
 *  RETURNS:	errNone				-- no error
 *				expErrNotOpen		-- FS driver library has not been opened
 *				vfsErrFileBadRef	-- the fileref is invalid
 *              vfsErrIsADirectory  -- the fileref points to a directory
 *				Sets fileSizeP to the size of the file.
 *
 *************************************************************/
Err FSFileSize(UInt16 fsLibRefNum, FileRef fileRef, UInt32 *fileSize)
{
	HostFSGlobalsPtr       gP;
	Err                    error = errNone;
	long                   tellPos;
	long                   fileLen;

	gP = SysLibTblEntry(fsLibRefNum)->globalsP;
	if (!gP)
		return expErrNotOpen;
		
	if(((HostFSFILE*)fileRef)->attributes & HOSTFLAG_ISDIR)
		return vfsErrIsADirectory;

	tellPos = HostFTell(((HostFSFILE*)fileRef)->hostFile);
	
	// Seek to end of file
	HostFSeek(((HostFSFILE*)fileRef)->hostFile, 0, 2);
	
	fileLen = HostFTell(((HostFSFILE*)fileRef)->hostFile);
	
	// Seek back to original location
	HostFSeek(((HostFSFILE*)fileRef)->hostFile, tellPos, 0);
	
	*fileSize = fileLen;
	
	return error;
}


/********************************************************************
 * Directory APIs:
 ********************************************************************/
 
/************************************************************
 *
 *  FUNCTION:	FSDirCreate
 *
 *  DESCRIPTION:	Create a new directory
 *					All parts of the path, except the last part must already exist.
 *
 *  PARAMETERS:	fsLibRefNum			-- FS library reference number
 *				volRefNum			-- Volume reference number passed to FSVolumeMount
 *				dirNameP			-- Full path of the directory to be created
 *
 *  RETURNS:	errNone					-- no error
 *				expErrNotOpen			-- FS driver library has not been opened
 *				vfsErrVolumeBadRef		-- the volume has not been mounted with FSVolumeMount
 *				vfsErrFileAlreadyExists	-- a directory or file with this name already exists at this location
 *				vfsErrBadName			-- the full path before the new directory does not exist
 *
 *************************************************************/
Err FSDirCreate(UInt16 fsLibRefNum, UInt16 volRefNum, const Char *dirNameP)
{
	HostFSGlobalsPtr gP;
	Err error = errNone;
	HostFSMountedVolumePtr mV;
	
	gP = SysLibTblEntry(fsLibRefNum)->globalsP;
	if (!gP)
		return expErrNotOpen;

	if(volRefNum == 0) return vfsErrVolumeBadRef;
		
	PrvFindMountedVolume(gP, volRefNum, &mV);
	if(mV == NULL)
		return vfsErrVolumeBadRef;

	DO_PATH_CONVERSION(dirNameP);

	error = HostMkDir(dirNameP);
	if(error != errNone) return PrvFSErrEquiv(error);

	return errNone;
}


/************************************************************
 *
 *  FUNCTION:	FSDirEntryEnumerate
 *
 *  DESCRIPTION:	Enumerate the entries in the given directory.
 *					Pass a dirEntryIteratorP of expIteratorStart to get the first entry.
 *					dirEntryIteratorP will be updated to the next item.
 *					When returning the last directory entry, the dirEntryIteratorP will be
 *					set to expIteratorStop.  If there are no directory entries to enumerate,
 *					FSDirEntryEnumerate will return expErrEnumerationEmpty when expIteratorStart is passed
 *					for the dirEntryIteratorP.
 *					The directory to be enumerated must first be opened in order
 *					to obtain a FileRef.
 *					This function is only permitted for FileRefs to open directories.
 *
 *					FSDirEntryEnumerate should be called like this:
 *						FileInfoType info;
 *						UInt32 dirIterator = expIteratorStart;
 *						while (dirIterator != expIteratorStop) {
 *							if ((err = FSDirEntryEnumerate(fsLibRefNum, dirRef, &dirIterator, &info)) != errNone) {
 *								// Do something with the directory entry info
 *							} else {
 *								// handle error... possibly by breaking out of the loop
 *							}
 *						}
 *
 *  PARAMETERS:	fsLibRefNum			-- FS library reference number
 *				dirRef				-- Directory reference number returned from FSFileOpen
 *				dirEntryIteratorP	-- Reference to the last entry enumerated
 *										Pass expIteratorStart to get the first entry
 *										This is updated on return to reference the next entry
 *										or set to expIteratorStop if infoP is the last directory entry.
 *				infoP				-- receives Info on the directory entry specified with dirEntryIteratorP
 *
 *  RETURNS:	errNone					-- no error, infoP is valid
 *				expErrNotOpen			-- FS driver library has not been opened
 *				vfsErrFileBadRef		-- the fileref is invalid
 *				vfsErrNotADirectory		-- the fileref is valid, but does not point to a directory entry
 *				sysErrParamErr			-- the dirEntryIteratorP is not valid
 *				expErrEnumerationEmpty	-- there are no directory entries left to enumerate
 *				Fills in infoP for the directory entry specified with dirEntryIteratorP
 *				Updates dirEntryIteratorP to point to the next entry in the directory,
 *				or sets it to expIteratorStop when returning the last directory entry.
 *
 *************************************************************/
Err FSDirEntryEnumerate(UInt16 fsLibRefNum, FileRef dirRef, UInt32 *dirEntryP, 
		FileInfoType *infoP)
{
	HostFSGlobalsPtr       gP;
	Err                    error = errNone;
	HostDIRType*           dirP;
	HostDirEntType         *entryP, *tmpEntryP;
	Int32                  ctr;
	HostStatType           stat;
	HostErr                hostErr;
	
	gP = SysLibTblEntry(fsLibRefNum)->globalsP;
	if (!gP)
		return expErrNotOpen;

	if(!(((HostFSFILE*)dirRef)->attributes & HOSTFLAG_ISDIR))
		return vfsErrNotADirectory;
	
	if(*dirEntryP == expIteratorStop)
	{
		return expErrEnumerationEmpty;
	}
	
	dirP = ((HostFSFILE*)dirRef)->hostDir;

	/* We need to read forward to the place the iterator tells us we should be in the directory.
	   There's probably some faster way to do this, but I don't know a good one that works as an
	   iterator should work. */
	
	/* Reset position */
	/* Don't bother to check for error on close -- what would we do if it happened? */
	HostCloseDir(dirP);
	dirP = ((HostFSFILE*)dirRef)->hostDir = HostOpenDir(((HostFSFILE*)dirRef)->filename);
	if(!dirP)
	{
		return PrvFSErrEquiv(HostErrNo());
	}
	
	// Now read forward by *dirEntryP entries
	for(ctr=0; ctr < *dirEntryP; ctr++)
	{
		tmpEntryP = HostReadDir(dirP);
		if(!tmpEntryP) break;
	}
	if(!tmpEntryP && (*dirEntryP > 0))
	{
		*dirEntryP = expIteratorStop;
		return expErrEnumerationEmpty;
	}

	// Now, finally read the entry proper
	entryP = HostReadDir(dirP);
	if(!entryP)
	{
		// This happens when we have no more entries
		// in the directory.  We'll just have to assume that that's
		// what happened and we're out of entries.
		*dirEntryP = expIteratorStop;
		return expErrEnumerationEmpty;
	}
	
	// Set name, if appropriate
	if(infoP->nameP)
	{
		StrNCopy(infoP->nameP, entryP->d_name, infoP->nameBufLen);
		if(StrLen(entryP->d_name) >= infoP->nameBufLen) {
			infoP->nameP[infoP->nameBufLen - 1] = '\0';
			return vfsErrBufferOverflow;
		}
	}

	/* Convert filename.  Note that the pathname in the file pointer is already converted. */
	StrCopy(gP->pathBuffer, ((HostFSFILE*)dirRef)->filename);
	/* Now append the filename to the buffer */
	{
		char sbuf[2];
		sbuf[0] = gP->separator;
		sbuf[1] = '\0';
		StrCat(gP->pathBuffer, sbuf);   // Append the dir separator character
	}
	StrCat(gP->pathBuffer, entryP->d_name);
	
	// Handle new value of *dirEntryP
	(*dirEntryP)++;
	tmpEntryP = HostReadDir(dirP);  // Read ahead to next entry
	if(!tmpEntryP)
	{
		// Final entry, normal read
		*dirEntryP = expIteratorStop;  // This is the last entry, so set the iterator that way, but no error
	}
	
	/* Now, we do attributes */	
	infoP->attributes = 0;
	hostErr = HostStat(gP->pathBuffer, &stat);
	if(hostErr == errNone)
	{
		if(stat.st_mode_ & 040000) {
			infoP->attributes |= vfsFileAttrDirectory;
		}
		if(stat.st_mode_ & 0120000)
			infoP->attributes |= vfsFileAttrLink;
		if(!(stat.st_mode_ & 0444))
			infoP->attributes |= vfsFileAttrReadOnly;
	}
	
	return errNone;
}



/********************************************************************
 * Volume APIs:
 ********************************************************************/

/********************************************************************
 * Enumerating all volumes:
 * 1. Call FSVolumeCount to get the total number of volumes
 * 2. volRefNums always start at 0, and increment from there
 * 3. For each volume, call FSVolumeGetLabel.  This will return fsErrVolumeNotMounted
 *		for unmounted volumes.
 *		Clients who need to be able to remount the volume should
 *		register for this notification, which contains information
 *		about the slot from which the volume was unmounted.
 *
 * Minns: may later add another call to make the enumeration of volumes
 *	more intuitive.
 ********************************************************************/

 
/************************************************************
 *
 *  FUNCTION:	FSVolumeFormat
 *
 *  DESCRIPTION:	Format the first volume on the specified slot
 *					(The Slot Driver currently only supports one volume per slot)
 *					A volume must not be mounted on the slot when calling this function.
 *
 *  LIBRARY DEVELOPER NOTES:
 *					As with all power-intensive operations, the filesystem
 *					should check to make sure that enough power is available
 *					to complete the format, and abort if not, returning
 *					expErrNotEnoughPower.
 *
 *					The vfsMountParamP should be switched off of
 *					vfsMountParamP->mountClass and cast to the correct
 *					MountParamType.  For example if mountClass == 'libs'
 *					then vfsMountParamP should be cast to VFSSlotMountParamType.
 *
 *					Before formatting the media, FSVolumeFormat should call
 *					SlotCardLowLevelFormat().  It should then call SlotCardMetrics()
 *					to get information about the physical metrics of the card.
 *					The actual format should be accomplished with SlotSectorWrite() calls.
 *
 *  PARAMETERS:	fsLibRefNum				-- FS library reference number
 *				vfsMountParamP			-- Mount parameters
 *
 *  RETURNS:	errNone					-- no error
 *				expErrNotOpen			-- FS driver library has not been opened
 *				expErrNotEnoughPower	-- the required power is not available
 *				vfsErrVolumeStillMounted	-- There is already a volume mounted on this slot
 *
 *************************************************************/
Err FSVolumeFormat(UInt16 fsLibRefNum, VFSAnyMountParamType *mountParam)
{
#pragma unused(fsLibRefNum)
#pragma unused(mountParam)
	
	ErrDisplay("You have called FSVolumeFormat on a POSE HostFS volume.  There are easier and much "
				"better ways to format your hard drive, so HostFS will not do so.");

	return expErrUnsupportedOperation;
}


/************************************************************
 *
 *  FUNCTION:	FSVolumeMount
 *
 *  DESCRIPTION:	Mount the first volume on the specified slot
 *					(The Slot Driver only supports one volume per slot)
 *					The VFS posts a sysNotifyVolumeMountedEvent notification once the filesystem
 *					is successfully mounted.
 *					VFSAnyMountParamPtr->volRefNum must be set on entry to the volRefNum that will
 *					be assigned to this volume if the mount succeeds.
 *
 *  LIBRARY DEVELOPER NOTES:
 *					The VFS will post the sysNotifyVolumeMountedEvent notification ONLY if this
 *					function returns errNone.
 *
 *					Before mounting, it is important to check to make sure
 *					that there is enough power to perform the mount.  If there is not
 *					enough power to mount, FSVolumeMount should return expErrNotEnoughPower
 *					and abort the mount operation.
 *
 *					The vfsMountParamP should be switched off of
 *					vfsMountParamP->mountClass and cast to the correct
 *					MountParamType.  For example if mountClass == 'libs'
 *					then vfsMountParamP should be cast to VFSSlotMountParamType.
 *
 *					The vfsMountParamP contains the global volRefNum that will be assigned
 *					to this volume if the mount succeeds.  The filesystem
 *					library should store this, as all other calls which take a volRefNum
 *					as a parameter will use this number.
 *
 *  PARAMETERS:	fsLibRefNum				-- FS library reference number
 *				vfsMountParamP			-- Mount parameters
 *
 *  RETURNS:	errNone					-- no error
 *				expErrNotOpen			-- FS driver library has not been opened
 *				expErrNotEnoughPower	-- the required power is not available
 *				vfsErrBadData			-- The volume cannot be mounted because the
 *											format of the media is unrecognized
 *											This will cause the VFS to attempt to format 
 *											and remount the media.
 *				memErrNotEnoughSpace    -- Not enough space to allocate required FS structures
 *
 *************************************************************/
Err FSVolumeMount(UInt16 fsLibRefNum, VFSAnyMountParamType *mountParam)
{
	HostFSGlobalsPtr gP;
	Err err = errNone;
	VFSPOSEMountParamType *poseMountP = (VFSPOSEMountParamType*)mountParam;
	HostFSMountedVolumePtr mV;
	
	gP = SysLibTblEntry(fsLibRefNum)->globalsP;
	if (!gP)
		return expErrNotOpen;
	
	if(mountParam->mountClass != vfsMountClass_POSE) 
		return expErrUnsupportedOperation;

	gP->poseCardVolRef[poseMountP->poseSlotNum] = mountParam->volRefNum;

	// Yes - volume is formatted, mount it:
	mV = MemPtrNew(sizeof(HostFSMountedVolumeType));
	if(!mV) return memErrNotEnoughSpace;
	MemPtrSetOwner(mV, 0);
	mV->volumeRef = mountParam->volRefNum;

	//  !!!!NOTE!!!! 
	//	If you get a compile error on this line, then you have
	// an outdated version of HostControl.h. Download the latest
	// version of the Palm OS Emulator and copy the HostControl.h
	// file included with the emulator to the Core/System include
	// directory of the SDK.
	//  !!!!!!!!!!! SEE ABOVE NOTE !!!!!!!!!!!!!!!!!!!!!!!!!!!!
	StrNCopy(mV->volprefix, HostSlotRoot(poseMountP->poseSlotNum), PATHPREFIXLEN);
	mV->volprefix[PATHPREFIXLEN - 1] = '\0';
	// Now remove dir separator character from end of SlotRoot
	{
		char *ind = mV->volprefix;
		while(*ind) ind++;  // Go to null-char at end
		ind--;
		if((*ind == ':') || (*ind == '\\') || (*ind == '/')) {
			*ind = '\0';
		}
	}
	mV->openFile = NULL;
	mV->next = gP->volume;
	mV->prev = NULL;
	
	if(mV->next) {
		mV->next->prev = mV;
	}
	
	gP->volume = mV;
	err = errNone;
	
	Exit:

	PrvVolumeVerify(gP);
	
	return err;
}


/************************************************************
 *
 *  FUNCTION:	FSVolumeUnmount
 *
 *  DESCRIPTION:	Unmount the given volume.  This closes any opened files.
 *					The VFS posts a sysNotifyVolumeUnmountedEvent notification once the filesystem
 *					is successfully unmounted.
 *
 *  LIBRARY DEVELOPER NOTES:
 *					The VFS will post the sysNotifyVolumeUnmountedEvent notification ONLY if this
 *					function returns errNone.
 *
 *  PARAMETERS:	fsLibRefNum				-- FS library reference number
 *				volRefNum				-- Volume reference number passed to FSVolumeMount
 *
 *  RETURNS:	errNone					-- no error
 *				expErrNotOpen			-- FS driver library has not been opened
 *				vfsErrVolumeBadRef		-- the volume has not been mounted with FSVolumeMount
 *
 *************************************************************/
Err FSVolumeUnmount(UInt16 fsLibRefNum, UInt16 volRefNum)
{
	HostFSGlobalsPtr       gP;
	Err                    err = errNone;
	HostFSMountedVolumePtr mV;
	int                    slotNum, ctr;

	gP = SysLibTblEntry(fsLibRefNum)->globalsP;
	if (!gP)
		return expErrNotOpen;
	
	if(volRefNum == 0) return vfsErrVolumeBadRef;
	
	PrvFindMountedVolume(gP, volRefNum, &mV);
	if(mV == NULL)
		return vfsErrVolumeBadRef;

	if(volRefNum != mV->volumeRef)
		return vfsErrVolumeBadRef;
	
	slotNum = -1;
	for(ctr=1; ctr<=gP->numSlots; ctr++) {
		if(gP->poseCardVolRef[ctr] == volRefNum) {
			slotNum = ctr; break;
		}
	}
	if(slotNum == -1) {
		ErrNonFatalDisplay("HostFSVolumeUnmount: Bad volref -- not found in any slot!");
		return vfsErrVolumeBadRef;
	}

	gP->poseCardVolRef[ctr] = 0;
	PrvCloseAllFiles(mV->openFile);
	mV->openFile = NULL;
	
	// Remove from mounted-volume list
	if(mV->prev) {
		mV->prev->next = mV->next;
	}
	if(mV->next) {
		mV->next->prev = mV->prev;
	}
	if(mV == gP->volume) gP->volume = mV->next;

	MemPtrFree(mV);
	
	PrvVolumeVerify(gP);
	
	return errNone;
}


/************************************************************
 *
 *  FUNCTION:	FSVolumeInfo
 *
 *  DESCRIPTION:	Get information about the specified volume
 *
 *  PARAMETERS:	fsLibRefNum				-- FS library reference number
 *				volRefNum				-- Volume reference number passed to FSVolumeMount
 *				volInfoP				-- Receives the volume info for the specified volume
 *
 *  RETURNS:	errNone					-- no error
 *				expErrNotOpen			-- FS driver library has not been opened
 *				vfsErrVolumeBadRef		-- the volume has not been mounted with FSVolumeMount
 *				volInfoP is filled in on return with information about the specified volume
 *
 *************************************************************/
Err FSVolumeInfo(UInt16 fsLibRefNum, UInt16 volRefNum, VolumeInfoType *volInfoP)
{
	HostFSGlobalsPtr       gP;
	Err                    err = errNone;
	HostFSMountedVolumePtr mV;

	gP = SysLibTblEntry(fsLibRefNum)->globalsP;
	if (!gP)
		return expErrNotOpen;
		
	PrvFindMountedVolume(gP, volRefNum, &mV);
	if(mV == NULL)
		return vfsErrVolumeBadRef;
	
	if(volRefNum != mV->volumeRef) 
		return vfsErrVolumeBadRef;
	
	volInfoP->attributes = 0;
	volInfoP->slotRefNum = 0;
	
	volInfoP->mediaType = expMediaType_PoserHost;
	volInfoP->fsCreator = kHostFSLibCreator;
	
	return err;
}


/************************************************************
 *
 *  FUNCTION:	FSVolumeGetLabel
 *
 *  DESCRIPTION:	Determine the volume label for a particular volume.
 *					Clients that wish to remember a particular volume
 *					should save this rather than the volRefNum. The volRefNum
 *					may be different each time the volume is mounted.
 *
 *					Volume labels are up to 255 characters long,
 *					using any normal character including spaces and lower case
 *					characters in any character set and the following special characters:
 *					$ % ' - _ @ ~ ` ! ( ) ^ # & + , ; = [ ]
 *
 *  PARAMETERS:	fsLibRefNum				-- FS library reference number
 *				volRefNum				-- Volume reference number passed to FSVolumeMount
 *				labelP					-- Receives the label of the volume.
 *				bufLen					-- The length of the labelP buffer
 *
 *  RETURNS:	errNone					-- no error
 *				expErrNotOpen			-- FS driver library has not been opened
 *				vfsErrVolumeBadRef		-- the volume has not been mounted with FSVolumeMount
 *				vfsErrBufferOverflow	-- bufLen is not big enough to receive the full volume label
 *				labelP is filled in on return with the volume label for the specified volume
 *
 *************************************************************/
Err FSVolumeGetLabel(UInt16 fsLibRefNum, UInt16 volRefNum, Char *outLabelP, UInt16 bufLen)
{
	HostFSGlobalsPtr gP;
	int ctr;
	int sN;
	
	gP = SysLibTblEntry(fsLibRefNum)->globalsP;
	if (!gP)
		return expErrNotOpen;
	
	if(volRefNum == 0) return vfsErrVolumeBadRef;
	
	StrNCopy(outLabelP, "POSESlot", bufLen - 3);
	outLabelP[bufLen-3] = '\0';
	
	sN = -1;
	for(ctr=0; ctr <= gP->numSlots; ctr++) {
		if(gP->poseCardVolRef[ctr] == volRefNum) {
			sN = ctr; break;
		}
	}
	if(sN == -1) {
		StrCat(outLabelP, "??");
	} else {
		char numBuf[2];
		numBuf[0] = '0' + sN;
		ErrNonFatalDisplayIf(sN > 9, "This isn't going to work for slot numbers above 9...");
		numBuf[1] = '\0';
		StrCat(outLabelP, numBuf);
	}
	return errNone;
}


/************************************************************
 *
 *  FUNCTION:	FSVolumeSetLabel
 *
 *  DESCRIPTION:	Change the volume label for a particular volume.
 *					Most clients should not need to call this routine.
 *
 *					Volume labels are up to 255 characters long,
 *					using any normal character including spaces and lower case
 *					characters in any character set and the following special characters:
 *					$ % ' - _ @ ~ ` ! ( ) ^ # & + , ; = [ ]
 *
 *  PARAMETERS:	fsLibRefNum				-- FS library reference number
 *				volRefNum				-- Volume reference number passed to FSVolumeMount
 *				labelP					-- Label to apply to the specified volume (NUL-terminated)
 *
 *  RETURNS:	errNone					-- no error
 *				expErrNotOpen			-- FS driver library has not been opened
 *				vfsErrVolumeBadRef		-- the volume has not been mounted with FSVolumeMount
 *
 *************************************************************/
Err FSVolumeSetLabel(UInt16 fsLibRefNum, UInt16 volRefNum, const Char *inLabelP)
{
#pragma unused(inLabelP)
	HostFSGlobalsPtr gP;
	
	gP = SysLibTblEntry(fsLibRefNum)->globalsP;
	if (!gP)
		return expErrNotOpen;
	
	if(volRefNum == 0) return vfsErrVolumeBadRef;
		
	return expErrUnimplemented;
}


/************************************************************
 *
 *  FUNCTION:	FSVolumeSize
 *
 *  DESCRIPTION:	Determine the total amount of space on a volume,
 *					and the amount that is currently used. (in bytes)
 *
 *  PARAMETERS:	fsLibRefNum				-- FS library reference number
 *				volRefNum				-- Volume reference number passed to FSVolumeMount
 *				volumeUsedP				-- Receives the amount of used space on the volume
 *				volumeTotalP			-- Receives the total amount of space on the volume
 *
 *  RETURNS:	errNone					-- no error
 *				expErrNotOpen			-- FS driver library has not been opened
 *				vfsErrVolumeBadRef		-- the volume has not been mounted with FSVolumeMount
 *				Sets volumeUsedP and volumeTotalP to the amount of used and total
 *				space on the volume.
 *
 *************************************************************/
Err FSVolumeSize(UInt16 fsLibRefNum, UInt16 volRefNum, UInt32 *volumeUsed, UInt32 *volumeTotal)
{
	HostFSGlobalsPtr       gP;
	Err                    error = errNone;
	HostFSMountedVolumePtr mV;

	gP = SysLibTblEntry(fsLibRefNum)->globalsP;
	if (!gP)
		return expErrNotOpen;
		
	PrvFindMountedVolume(gP, volRefNum, &mV);
	if(mV == NULL)
		return vfsErrVolumeBadRef;
	
	if(volRefNum != mV->volumeRef) 
		return vfsErrVolumeBadRef;
	
	*volumeUsed = 1024;          // Dummy value

	*volumeTotal = ((UInt32)100 * 1024);  // Dummy value
	
	return error;
}



/************************************************************
 *
 *  FUNCTION:	PrvFindMountedVolume
 *
 *  DESCRIPTION:  Finds the global HostFSMountedVolumeType instance that matches a particular VolumeRef
 *                in the global volume list.
 *
 *  PARAMETERS:	gP - the HostFS globals pointer
 *              volumeRef - the volumeRef to match with an entry from the list
 *              mountedVolumeP - a pointer to the volume pointer, used to return the MountedVolumePtr
 *
 *  CALLED BY:	FS functions like FSFileOpen that take a volumeRef
 *
 *  RETURNS:	errNone				-- no error
 *				anything else		-- Error occurred finding volume (volume not in global list)
 *
 *************************************************************/ 
Err PrvFindMountedVolume(HostFSGlobalsPtr gP, UInt16 volumeRef, HostFSMountedVolumePtr* mountedVolumeP)
{
	HostFSMountedVolumePtr index = gP->volume;
	
	*mountedVolumeP = NULL;
	
	while(index)
	{
		if(volumeRef == index->volumeRef)
		{
			*mountedVolumeP = index;
			return errNone;
		}
		index = index->next;
	}
	return -1;
}


/************************************************************
 *
 *  FUNCTION:	PrvFSPathConvert
 *
 *  DESCRIPTION: Called by functions to convert the path into something the HostControl API understands.
 *
 *  PARAMETERS:	oldPath - The path to be converted
 *              newPath - An allocated buffer to hold the new, converted path
 *              separator - The directory separator to use in the new, converted path
 *
 *  CALLED BY:	FS functions like FSFileOpen and FSFileSizeGet
 *
 *  RETURNS:	errNone				-- no error
 *				anything else		-- Error occurred converting path (should never happen)
 *
 *************************************************************/ 
Err PrvFSPathConvert(const Char *oldPath, Char* newPath, Char separator, Char* pathPrefix, Char* volPrefix)
{
	char prevWasSep;  // Was the last character processed a directory separator?
	
	StrCopy(newPath, pathPrefix);
	StrCat(newPath, volPrefix);
	while(*newPath) newPath++;   // Advance to end of string
	prevWasSep = 0;
	while(*oldPath)
	{
		if(*oldPath == '/' || *oldPath == '\\')
		{
			if(!prevWasSep) {
				*newPath = separator;
			}
			prevWasSep = 1;
		}
		else
		{
			*newPath = *oldPath;
			prevWasSep = 0;
		}
		newPath++;
		oldPath++;
	}
	if(prevWasSep && (StrLen(newPath) > 1)) {
		newPath--;  // Remove separator character at the end
	}
	*newPath = '\0';
	return errNone;
}


/************************************************************
 *
 *  FUNCTION:	PrvFindFilename
 *
 *  DESCRIPTION: Finds the final directory separator of a pathname.
 *
 *  PARAMETERS:	oldPath - The path to be converted
 *              newPath - An allocated buffer to hold the new, converted path
 *              separator - The directory separator to use in the new, converted path
 *
 *  CALLED BY:	FSFileRename
 *
 *  RETURNS:	errNone				-- no error
 *				anything else		-- Error occurred converting path (should never happen)
 *
 *************************************************************/ 
Char* PrvFindFilename(Char *path)
{
	Char* lastSep;
	
	lastSep = path;
	while(*path)
	{
		if(*path == '/' || *path == ':' || *path == '\\') lastSep = path;
		path++;
	}
	lastSep++;    // Advance to character after dir-separator character
	return lastSep;
}


/************************************************************
 *
 *  FUNCTION:	PrvFSErrEquiv
 *
 *  DESCRIPTION:	Function to translate HostControl file error codes to PalmOS
 *              	error codes.  Takes the HostControl error code and returns the
 *              	translated one.
 *
 *  PARAMETERS:	hostError -- An error returned or returnable by HostControl
 *
 *  CALLED BY:	Any FS function which can receive an error from HostControl.
 *
 *  RETURNS:	any valid FSLib, VFS, Slot, or system-general error, or -1 for an unrecognized
 *              or not-yet-translatable error.
 *
 *************************************************************/ 
Err PrvFSErrEquiv(HostErr hE)
{
	// If I put these together in a single large case statement, CodeWarrior converts it to a function
	// call and gives me a link error;  I'd have to do things I don't want to do to the project file
	// right now to fix it.  Anything more than four cases per switch causes the problem.
	switch(hE) {
		case hostErrNone: return errNone;
		case hostErrFileNotFound: return vfsErrFileNotFound;
		case hostErrDiskError: return vfsErrFileGeneric;
		case hostErrInvalidParameter: return sysErrParamErr;
	}
	switch(hE) {
		case hostErrMemReadOutOfRange: return sysErrParamErr;
		case hostErrMemWriteOutOfRange: return sysErrParamErr;
		case hostErrOutOfMemory: return memErrNotEnoughSpace;
		case hostErrFileNameTooLong: return vfsErrBadName;
	}
	switch(hE) {
		case hostErrPermissions: return vfsErrFilePermissionDenied;
		case hostErrNotADirectory: return vfsErrNotADirectory;
		case hostErrTooManyFiles: return vfsErrVolumeFull;
		case hostErrReadOnlyFS: return expErrCardReadOnly;
	}
	switch(hE) {
		case hostErrIsDirectory: return vfsErrIsADirectory;
		case hostErrExists: return vfsErrFileAlreadyExists;
		case hostErrOpNotAvailable: return expErrUnimplemented;
		case hostErrDirNotEmpty: return vfsErrDirNotEmpty;
	}
	switch(hE) {
		case hostErrDiskFull: return vfsErrVolumeFull;
		case hostErrUnknownError: return vfsErrFileGeneric;

		default:
			return vfsErrFileGeneric;
	}
}


/************************************************************
 *
 *  FUNCTION:	PrvCloseAllFiles
 *
 *  DESCRIPTION:	Takes a list of HostFS file structures and closes all the files in the list.
 *					Used to close all open files in a volume when it's unmounted.
 *
 *  PARAMETERS:	openFile - the file-list (which is the same as an open file, really) to close.
 *
 *  CALLED BY:	FSVolumeUnmount
 *
 *  RETURNS:    errNone for no error, or any error that can be returned by VFSFileClose
 *
 *************************************************************/ 
Err PrvCloseAllFiles(HostFSFILE* openFile)
{
	HostFSFILE* index;
	Err         returnerr = errNone;
	HostFSFILE* tmp;
	Err         err;
	
	index = openFile;
	while(index)
	{
		tmp = index;
		index = index->next;
		err = VFSFileClose((FileRef)tmp);
		if(err != errNone) returnerr = err;
	}
	
	return returnerr;
}


/************************************************************
 *
 *  FUNCTION:	PrvMountPOSEVolumes
 *
 *  DESCRIPTION:	Iterates through the available POSE slots, and mounts any that POSE claims contain
 *					virtual cards.
 *
 *  PARAMETERS:	gP - HostFS global variable pointer
 *
 *  CALLED BY:	FSLibOpen
 *
 *  RETURNS:	nothing
 *
 *************************************************************/ 
static void PrvMountPOSEVolumes(HostFSGlobalsPtr gP)
{
	long numSlots;
	long ctr;
	Err  error;

	//  !!!!NOTE!!!! 
	//	If you get a compile error on this line, then you have
	// an outdated version of HostControl.h. Download the latest
	// version of the Palm OS Emulator and copy the HostControl.h
	// file included with the emulator to the Core/System include
	// directory of the SDK.
	//  !!!!!!!!!!! SEE ABOVE NOTE !!!!!!!!!!!!!!!!!!!!!!!!!!!!
	numSlots = HostSlotMax();
	gP->numSlots = numSlots;
	ErrNonFatalDisplayIf(numSlots > HOSTFS_MAX_SLOTS, "HostFS not configured for full number of slots, ignoring extras");
	if(numSlots > HOSTFS_MAX_SLOTS) numSlots = HOSTFS_MAX_SLOTS;

	for(ctr=1; ctr<=numSlots; ctr++) {
		VFSPOSEMountParamType mountParam;

	//  !!!!NOTE!!!! 
	//	If you get a compile error on this line, then you have
	// an outdated version of HostControl.h. Download the latest
	// version of the Palm OS Emulator and copy the HostControl.h
	// file included with the emulator to the Core/System include
	// directory of the SDK.
	//  !!!!!!!!!!! SEE ABOVE NOTE !!!!!!!!!!!!!!!!!!!!!!!!!!!!
		if(!HostSlotHasCard(ctr)) continue;  // If slot is empty, ignore it

		// Otherwise, card in slot -- mount it.
		mountParam.vfsMountParam.mountClass = vfsMountClass_POSE;
		mountParam.poseSlotNum = ctr;

		error = VFSVolumeMount(0, 0, (VFSAnyMountParamType*)&mountParam);
		if(error != errNone) {
			//ErrNonFatalDisplay("Error mounting a HostFS volume -- skipping mounting further volumes");
			break;
		}
	}
}

/************************************************************
 *
 *  FUNCTION:	PrvUnmountPOSEVolumes
 *
 *  DESCRIPTION:	Unmounts all mounted POSE volumes
 *
 *  PARAMETERS:	gP - HostFS global variable pointer
 *
 *  CALLED BY:	FSLibClose
 *
 *  RETURNS:	nothing
 *
 *************************************************************/ 
static void PrvUnmountPOSEVolumes(HostFSGlobalsPtr gP) {
	int ctr = 0;

	for(ctr = 0; ctr < gP->numSlots; ctr++) {
		if(gP->poseCardVolRef[ctr]) {
			VFSVolumeUnmount(gP->poseCardVolRef[ctr]);
		}
	}
}

/************************************************************
 *
 *  FUNCTION:	PrvAdjustNumSlots
 *
 *  DESCRIPTION:	Adds or subtracts POSE slots, empty or full.  Unmounts removed volumes.  Adjusts
 *					gP->numSlots for new number.
 *
 *  PARAMETERS:	gP - HostFS global variable pointer
 *
 *  CALLED BY:	FSCustomControl when a slot is mounted or unmounted
 *
 *  RETURNS:	nothing
 *
 *************************************************************/ 
void PrvAdjustNumSlots(HostFSGlobalsPtr gP) {
	int numSlots, ctr;
	
	numSlots = HostSlotMax();
	if(numSlots > HOSTFS_MAX_SLOTS) numSlots = HOSTFS_MAX_SLOTS;
	if(gP->numSlots != numSlots) {
		ErrNonFatalDisplay("PrvAdjustNumSlots: modified number of slots!");
				
		if(gP->numSlots > numSlots) {
			// Unmount the excess...
			for(ctr = gP->numSlots; ctr < numSlots; ctr++) {
				if(gP->poseCardVolRef[ctr]) {
					VFSVolumeUnmount(gP->poseCardVolRef[ctr]);
				}
			}
		}
		gP->numSlots = numSlots;
	}
}

/************************************************************
 *
 *  FUNCTION:	PrvResetNotifyProc
 *
 *  DESCRIPTION:	Called at reset-finished to mount the initially-mounted POSE volumes and register for
 *					the notification the later mounts and unmounts will cause.
 *
 *  PARAMETERS:	notifyParamsP - The parameters to the notification.  Mostly, just the globals pointer.
 *
 *  CALLED BY:	UIAppShell(?) when reset is done.
 *
 *  RETURNS:	errNone always
 *
 *************************************************************/ 
static Err PrvResetNotifyProc(SysNotifyParamType* notifyParamsP)
{
	HostFSGlobalsPtr gP;
	Err              error;

	gP = (HostFSGlobalsPtr)notifyParamsP->userDataP;
	PrvMountPOSEVolumes(gP);
	error = SysNotifyRegister(0, sysNotifyNoDatabaseID, sysNotifyPOSEMountEvent,
								PrvMountNotifyProc, sysNotifyNormalPriority, (void*)gP);
	ErrNonFatalDisplayIf(error != errNone,
						"Register failed, runtime slot mount/unmount disabled until next reset");
	return errNone;
}

/************************************************************
 *
 *  FUNCTION:	PrvMountNotifyProc
 *
 *  DESCRIPTION:	Called as a deferred notification when a POSE slot is mounted or unmounted.
 *
 *  PARAMETERS:	notifyParamsP - The parameters to the notification.  The globals pointer and details of the
 *				mount or unmount that this notification signals.
 *
 *  CALLED BY:	Deferred notification sent by FSCustomControl
 *
 *  RETURNS:	errNone always
 *
 *************************************************************/ 
static Err PrvMountNotifyProc(SysNotifyParamType* notifyParamsP)
{
	UInt16 slotNum;
	VFSPOSEMountParamType mountParam;
	HostFSGlobalsPtr gP;
	Err error;
	
	gP = (HostFSGlobalsPtr)notifyParamsP->userDataP;
	slotNum = (UInt16)notifyParamsP->notifyDetailsP & 0xFFFF;
	if(((UInt32)notifyParamsP->notifyDetailsP) & 0x10000) {
		// Mount request

		mountParam.vfsMountParam.mountClass = vfsMountClass_POSE;
		mountParam.poseSlotNum = slotNum;
		error = VFSVolumeMount(0, 0, (VFSAnyMountParamPtr)&mountParam);
	} else {
		// Unmount request
		error = VFSVolumeUnmount(gP->poseCardVolRef[slotNum]);
	}
	//ErrNonFatalDisplayIf(error != errNone, "Error doing dynamic mount/unmount!");
	return errNone;
}




void PrvVolumeVerify(HostFSGlobalsPtr gP) {
	HostFSMountedVolumePtr mV, prev;
	int                    sN, ctr;
	
	prev = NULL;
	mV = gP->volume;
	while(mV) {
		ErrFatalDisplayIf(prev != mV->prev, "Prev pointer incorrect in volume list!");
		
		// Check to see if mV is in mounted volumes by slot
		sN = -1;
		for(ctr = 1; ctr <= gP->numSlots; ctr++) {
			if(gP->poseCardVolRef[ctr] == mV->volumeRef) {
				sN = ctr;
				break;
			}
		}
		ErrFatalDisplayIf(sN == -1, "Mounted volume not in any slot in PrvVolumeVerify!");
		
		prev = mV;
		mV = mV->next;
	}
}

/************************************************************
 *
 *  FUNCTION:	PrvCopyBool
 *
 *  DESCRIPTION:	Copy data to a ptr that might be in the dynamic or storage heaps.
 *					
 *
 *  PARAMETERS:	bufBaseP				-- ptr to destination chunk
 *				offset					-- Offset within chunk to write data
 *				srcP					-- Ptr to source data
 *				numBytes				-- number of bytes of source data to copy
 *              DataStoreBased          -- whether bufBaseP is in the data store
 *
 *  RETURNS:	errNone if no error
 *
 *************************************************************/

Err PrvCopyBool(void* bufBaseP, UInt32 offset, void *srcP, UInt32 numBytes, Boolean DataStoreBased)
{
	Err err = errNone;
	
	if(DataStoreBased)
	{
		err = DmWrite(bufBaseP, offset, srcP, numBytes);
	}
	else
	{
		err = MemMove(((UInt8*)bufBaseP)+offset, srcP, numBytes);
	}
	
	return err;
}
