/* -*- mode: C++; tab-width: 4 -*- */
/* ===================================================================== *\
	Copyright (c) 2000-2001 Palm, Inc. or its subsidiaries.
	All rights reserved.

	This file is part of the Palm OS Emulator.

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.
\* ===================================================================== */

#include "EmCommon.h"
#include "EmTRGDiskType.h"


//---------------------------------------------------------------------------
// This file is used to determine what sort of memory card is being
// emulated ... so far, the only supported type is a generic 8MB card.
// Card emulation requires having a copy of the Tuple, Drive ID, and
// format sectors for a particular card
//---------------------------------------------------------------------------

#define B(x)	x, 0
 
static const EmSector _8MBTuple = {{
 B(0x01), B(0x04), B(0xdf), B(0x4a), B(0x01), B(0xff), B(0x1c), B(0x04),
 B(0x02), B(0xd9), B(0x01), B(0xff), B(0x18), B(0x02), B(0xdf), B(0x01),
 B(0x20), B(0x04), B(0x07), B(0x00), B(0x00), B(0x00), B(0x15), B(0x15),
 B(0x04), B(0x01), B(0x55), B(0x4e), B(0x49), B(0x54), B(0x59), B(0x20),
 B(0x44), B(0x49), B(0x47), B(0x49), B(0x54), B(0x41), B(0x4c), B(0x00),
 B(0x33), B(0x2e), B(0x30), B(0x00), B(0xff), B(0x21), B(0x02), B(0x04),
 B(0x01), B(0x22), B(0x02), B(0x01), B(0x01), B(0x22), B(0x03), B(0x02),
 B(0x0c), B(0x0f), B(0x1a), B(0x05), B(0x01), B(0x03), B(0x00), B(0x02),
 B(0x0f), B(0x1b), B(0x08), B(0xc0), B(0xc0), B(0xa1), B(0x01), B(0x55),
 B(0x08), B(0x00), B(0x20), B(0x1b), B(0x06), B(0x00), B(0x01), B(0x21),
 B(0xb5), B(0x1e), B(0x4d), B(0x1b), B(0x0a), B(0xc1), B(0x41), B(0x99),
 B(0x01), B(0x55), B(0x64), B(0xf0), B(0xff), B(0xff), B(0x20), B(0x1b),
 B(0x06), B(0x01), B(0x01), B(0x21), B(0xb5), B(0x1e), B(0x4d), B(0x1b),
 B(0x0f), B(0xc2), B(0x41), B(0x99), B(0x01), B(0x55), B(0xea), B(0x61),
 B(0xf0), B(0x01), B(0x07), B(0xf6), B(0x03), B(0x01), B(0xee), B(0x20),
 B(0x1b), B(0x06), B(0x02), B(0x01), B(0x21), B(0xb5), B(0x1e), B(0x4d),
 B(0x1b), B(0x0f), B(0xc3), B(0x41), B(0x99), B(0x01), B(0x55), B(0xea),
 B(0x61), B(0x70), B(0x01), B(0x07), B(0x76), B(0x03), B(0x01), B(0xee),
 B(0x20), B(0x1b), B(0x06), B(0x03), B(0x01), B(0x21), B(0xb5), B(0x1e),
 B(0x4d), B(0x14), B(0x00), B(0xff), B(0xff), B(0xff), B(0xff), B(0xff),
 B(0xff), B(0xff), B(0xff), B(0xff), B(0xff), B(0xff), B(0xff), B(0xff),
 B(0xff), B(0xff), B(0xff), B(0xff), B(0xff), B(0xff), B(0xff), B(0xff),
 B(0xff), B(0xff), B(0xff), B(0xff), B(0xff), B(0xff), B(0xff), B(0xff),
 B(0xff), B(0xff), B(0xff), B(0xff), B(0xff), B(0xff), B(0xff), B(0xff),
 B(0xff), B(0xff), B(0xff), B(0xff), B(0xff), B(0xff), B(0xff), B(0xff),
 B(0xff), B(0xff), B(0xff), B(0xff), B(0xff), B(0xff), B(0xff), B(0xff),
 B(0xff), B(0xff), B(0xff), B(0xff), B(0xff), B(0xff), B(0xff), B(0xff),
 B(0xff), B(0xff), B(0xff), B(0xff), B(0xff), B(0xff), B(0xff), B(0xff),
 B(0xff), B(0xff), B(0xff), B(0xff), B(0xff), B(0xff), B(0xff), B(0xff),
 B(0xff), B(0xff), B(0xff), B(0xff), B(0xff), B(0xff), B(0xff), B(0xff),
 B(0xff), B(0xff), B(0xff), B(0xff), B(0xff), B(0xff), B(0xff), B(0xff),
 B(0xff), B(0xff), B(0xff), B(0xff), B(0xff), B(0xff), B(0xff), B(0xff),
}};

#define W(x, y)  x, y

static const EmSector _8MBDriveID =
{{
 W(0x8A, 0x84), W(0xF6, 0x00), W(0x00, 0x00), W(0x02, 0x00), W(0x00, 0x00), W(0x00, 0x02), W(0x20, 0x00), W(0x00, 0x00),
 W(0x80, 0x3D), W(0x00, 0x00), W(0x20, 0x20), W(0x20, 0x20), W(0x20, 0x20), W(0x20, 0x20), W(0x20, 0x20), W(0x20, 0x20),
 W(0x20, 0x20), W(0x20, 0x20), W(0x20, 0x20), W(0x20, 0x20), W(0x02, 0x00), W(0x02, 0x00), W(0x04, 0x00), W(0x65, 0x52),
 W(0x20, 0x76), W(0x2E, 0x31), W(0x32, 0x30), W(0x69, 0x48), W(0x61, 0x74), W(0x68, 0x63), W(0x20, 0x69), W(0x56, 0x43),
 W(0x36, 0x20), W(0x31, 0x2E), W(0x32, 0x2E), W(0x20, 0x20), W(0x20, 0x20), W(0x20, 0x20), W(0x20, 0x20), W(0x20, 0x20),
 W(0x20, 0x20), W(0x20, 0x20), W(0x20, 0x20), W(0x20, 0x20), W(0x20, 0x20), W(0x20, 0x20), W(0x20, 0x20), W(0x01, 0x00),
 W(0x00, 0x00), W(0x00, 0x02), W(0x00, 0x00), W(0x00, 0x01), W(0x00, 0x00), W(0x01, 0x00), W(0xF6, 0x00), W(0x02, 0x00),
 W(0x20, 0x00), W(0x80, 0x3D), W(0x00, 0x00), W(0x00, 0x01), W(0x80, 0x3D), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00),
 W(0x20, 0x00), W(0x80, 0x3D), W(0x00, 0x00), W(0x00, 0x01), W(0x80, 0x3D), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00),
 W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00),
 W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00),
 W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00),
 W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00),
 W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00),
 W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00),
 W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00),
 W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00),
 W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00),
 W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00),
 W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00),
 W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00),
 W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00),
 W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00),
 W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00),
 W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00),
 W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00),
 W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00),
 W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00),
 W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00),
 W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00),
 W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00),
 W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00), W(0x00, 0x00)
}};

static const EmSector _8MBMasterBootRecord =
{{
0xFA, 0x33, 0xC0, 0x8E, 0xD0, 0xBC, 0x00, 0x7C, 0x8B, 0xF4, 0x50, 0x07, 0x50, 0x1F, 0xFB, 0xFC, 
0xBF, 0x00, 0x06, 0xB9, 0x00, 0x01, 0xF2, 0xA5, 0xEA, 0x1D, 0x06, 0x00, 0x00, 0xBE, 0xBE, 0x07, 
0xB3, 0x04, 0x80, 0x3C, 0x80, 0x74, 0x0E, 0x80, 0x3C, 0x00, 0x75, 0x1C, 0x83, 0xC6, 0x10, 0xFE, 
0xCB, 0x75, 0xEF, 0xCD, 0x18, 0x8B, 0x14, 0x8B, 0x4C, 0x02, 0x8B, 0xEE, 0x83, 0xC6, 0x10, 0xFE, 
0xCB, 0x74, 0x1A, 0x80, 0x3C, 0x00, 0x74, 0xF4, 0xBE, 0x8B, 0x06, 0xAC, 0x3C, 0x00, 0x74, 0x0B, 
0x56, 0xBB, 0x07, 0x00, 0xB4, 0x0E, 0xCD, 0x10, 0x5E, 0xEB, 0xF0, 0xEB, 0xFE, 0xBF, 0x05, 0x00, 
0x4F, 0x75, 0xED, 0xBE, 0xA3, 0x06, 0xEB, 0xD3, 0xBE, 0xC2, 0x06, 0xBF, 0xFE, 0x7D, 0x81, 0x3D, 
0x55, 0xAA, 0x75, 0xC7, 0x8B, 0xF5, 0xEA, 0x00, 0x7C, 0x00, 0x00, 0x49, 0x6E, 0x76, 0x61, 0x6C, 
0x69, 0x64, 0x20, 0x70, 0x61, 0x72, 0x74, 0x69, 0x74, 0x69, 0x6F, 0x6E, 0x20, 0x74, 0x61, 0x62, 
0x6C, 0x65, 0x00, 0x45, 0x72, 0x72, 0x6F, 0x72, 0x20, 0x6C, 0x6F, 0x61, 0x64, 0x69, 0x6E, 0x67, 
0x20, 0x6F, 0x70, 0x65, 0x72, 0x61, 0x74, 0x69, 0x6E, 0x67, 0x20, 0x73, 0x79, 0x73, 0x74, 0x65, 
0x6D, 0x00, 0x4D, 0x69, 0x73, 0x73, 0x69, 0x6E, 0x67, 0x20, 0x6F, 0x70, 0x65, 0x72, 0x61, 0x74, 
0x69, 0x6E, 0x67, 0x20, 0x73, 0x79, 0x73, 0x74, 0x65, 0x6D, 0x89, 0x70, 0x61, 0x72, 0x74, 0x5F, 
0x62, 0x6F, 0x6F, 0x74, 0x00, 0x00, 0xEB, 0x3C, 0x90, 0x4D, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x01, 
0x01, 0x00, 0x01, 0x01, 0x20, 0xF4, 0x20, 0x00, 0x00, 0x00, 0x20, 0x3D, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x55, 0xAA
}};

static const EmSector _8MBBootRecord =
{{
0xEB, 0x3C, 0x90, 0x4D, 0x53, 0x44, 0x4F, 0x53, 0x35, 0x2E, 0x30, 0x00, 0x02, 0x08, 0x01, 0x00, 
0x02, 0x00, 0x02, 0x20, 0x3D, 0xF8, 0x06, 0x00, 0x20, 0x00, 0x02, 0x00, 0x20, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x29, 0x0D, 0x0B, 0x00, 0x00, 0x53, 0x41, 0x4E, 0x56, 0x4F, 
0x4C, 0x20, 0x20, 0x20, 0x20, 0x20, 0x46, 0x41, 0x54, 0x31, 0x32, 0x20, 0x20, 0x20, 0x00, 0x8E, 
0xC0, 0x8E, 0xD8, 0xFB, 0xFC, 0xBE, 0x6A, 0x00, 0x81, 0xC6, 0x00, 0x7C, 0xAC, 0x0A, 0xC0, 0x74, 
0x0A, 0xB4, 0x0E, 0xB7, 0x00, 0xB3, 0x07, 0xCD, 0x10, 0xEB, 0xF1, 0x33, 0xC0, 0xCD, 0x16, 0xCD, 
0x19, 0xBE, 0xB3, 0x00, 0x81, 0xC6, 0x00, 0x7C, 0xEB, 0xE2, 0x0D, 0x0A, 0x4E, 0x6F, 0x6E, 0x2D, 
0x53, 0x79, 0x73, 0x74, 0x65, 0x6D, 0x20, 0x64, 0x69, 0x73, 0x6B, 0x20, 0x6F, 0x72, 0x20, 0x64, 
0x69, 0x73, 0x6B, 0x20, 0x65, 0x72, 0x72, 0x6F, 0x72, 0x0D, 0x0A, 0x52, 0x65, 0x70, 0x6C, 0x61, 
0x63, 0x65, 0x20, 0x61, 0x6E, 0x64, 0x20, 0x73, 0x74, 0x72, 0x69, 0x6B, 0x65, 0x20, 0x61, 0x6E, 
0x79, 0x20, 0x6B, 0x65, 0x79, 0x20, 0x77, 0x68, 0x65, 0x6E, 0x20, 0x72, 0x65, 0x61, 0x64, 0x79, 
0x0D, 0x0A, 0x00, 0x0D, 0x0A, 0x44, 0x69, 0x73, 0x6B, 0x20, 0x42, 0x6F, 0x6F, 0x74, 0x20, 0x66, 
0x61, 0x69, 0x6C, 0x75, 0x72, 0x65, 0x0D, 0x0A, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x55, 0xAA
}};

static const EmSector _8MBFat = 
{{
0xF8, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
}};

static const EmSector _8MBRootDir = 
{{
0xE5, 0xF0, 0xE1, 0xD2, 0xC3, 0xB4, 0xA5, 0x96, 0x87,  'V',  '?', 0x28, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x83, 0x86, 0x54, 0x29, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
 'T',  'R',  'G',  'P',  'R',  'O',  '_',  '8',  'M',  'B', 0x20, 0x28, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x83, 0x86, 0x54, 0x29, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
}};


// ---------------------------------------------------------------------------
//		 EmGeneric8MB::EmGeneric8MB
// ---------------------------------------------------------------------------
EmGeneric8MB::EmGeneric8MB(void)
{
}

// ---------------------------------------------------------------------------
//		 EmGeneric8MB::~EmGeneric8MB
// ---------------------------------------------------------------------------
EmGeneric8MB::~EmGeneric8MB(void)
{
}

// ---------------------------------------------------------------------------
//		 EmGeneric8MB::GetTuple
// ---------------------------------------------------------------------------
EmSector * EmGeneric8MB::GetTuple(void)
{
	return((EmSector *)&_8MBTuple);
}

// ---------------------------------------------------------------------------
//		 EmGeneric8MB::GetDriveID
// ---------------------------------------------------------------------------
EmSector * EmGeneric8MB::GetDriveID(void)
{
	return((EmSector *)&_8MBDriveID);
}

#define	_8MB_NUM_SECTORS		16384

// ---------------------------------------------------------------------------
//		 EmGeneric8MB::GetNumSectors
// ---------------------------------------------------------------------------
uint32 EmGeneric8MB::GetNumSectors(void)
{
	return _8MB_NUM_SECTORS;
}

// ---------------------------------------------------------------------------
//		 EmGeneric8MB::GetSector
// ---------------------------------------------------------------------------
EmSector * EmGeneric8MB::GetSector(LogicalBlockAddr lba)
{
	switch(lba)
	{
	case 0 :
		return((EmSector *)&_8MBMasterBootRecord);
	case 32 :
		return((EmSector *)&_8MBBootRecord);
	case 33 :
	case 39 :
		return((EmSector *)&_8MBFat);
	case 45 :
		return((EmSector *)&_8MBRootDir);
	default :
		return(EmDiskType::GetEmptySector());
	}
}

static const EmSector DummySector =
{{
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
}};

// ---------------------------------------------------------------------------
//		 EmDiskType::EmDiskType
// ---------------------------------------------------------------------------
EmDiskType::EmDiskType(void)
{
}


// ---------------------------------------------------------------------------
//		 EmDiskType::~EmDiskType
// ---------------------------------------------------------------------------
EmDiskType::~EmDiskType(void)
{
}

// ---------------------------------------------------------------------------
//		 EmDiskType::GetTuple
// ---------------------------------------------------------------------------
EmSector * EmDiskType::GetTuple(void)
{
	return (EmSector *)&DummySector;
}

// ---------------------------------------------------------------------------
//		 EmDiskType::GetDriveID
// ---------------------------------------------------------------------------
EmSector * EmDiskType::GetDriveID(void)
{
	return (EmSector *)&DummySector;
}

// ---------------------------------------------------------------------------
//		 EmDiskType::GetEmptySector
// ---------------------------------------------------------------------------
EmSector * EmDiskType::GetEmptySector(void)
{
	return (EmSector *)&DummySector;
}

// ---------------------------------------------------------------------------
//		 EmCurrDiskType::EmCurrDiskType
// ---------------------------------------------------------------------------
EmCurrDiskType::EmCurrDiskType (void)
{
}

// ---------------------------------------------------------------------------
//		 EmCurrDiskType::~EmCurrDiskType
// ---------------------------------------------------------------------------
EmCurrDiskType::~EmCurrDiskType(void)
{
}


// ---------------------------------------------------------------------------
//		 EmCurrDiskType::GetTuple
// ---------------------------------------------------------------------------
void EmCurrDiskType::GetTuple(EmDiskTypeID ID,
                              EmSector *   s)
{
	switch(ID)
	{
		case EM_DISK_GENERIC_8MB :
			*s = *Generic8MB.GetTuple();
			break;
		default :
			*s = *UnknownDisk.GetTuple();
			break;
	}
}

// ---------------------------------------------------------------------------
//		 EmCurrDiskType::GetDriveID
// ---------------------------------------------------------------------------
void EmCurrDiskType::GetDriveID(EmDiskTypeID ID,
                                EmSector *   s)
{
	switch(ID)
	{
		case EM_DISK_GENERIC_8MB :
			*s = *Generic8MB.GetDriveID();
			break;
		default :
			*s = *UnknownDisk.GetDriveID();
			break;
	}
}

// ---------------------------------------------------------------------------
//		 EmCurrDiskType::GetNumSectors
// ---------------------------------------------------------------------------
uint32 EmCurrDiskType::GetNumSectors(EmDiskTypeID ID)
{
	switch(ID)
	{
		case EM_DISK_GENERIC_8MB :
			return Generic8MB.GetNumSectors();
		default :
			return 0;
	}
}

// ---------------------------------------------------------------------------
//		 EmCurrDiskType::GetSector
// ---------------------------------------------------------------------------
void EmCurrDiskType::GetSector(EmDiskTypeID     ID,
                               LogicalBlockAddr lba,
                               EmSector *       s)
{
	switch(ID)
	{
		case EM_DISK_GENERIC_8MB :
			*s = *Generic8MB.GetSector(lba);
			break;
		default :
			*s = *UnknownDisk.GetEmptySector();
			break;
	}
}
