/* -*- mode: C++; tab-width: 4 -*- */
/* ===================================================================== *\
	Copyright (c) 1998-2000 Palm, Inc. or its subsidiaries.
	All rights reserved.

	This file is part of the Palm OS Emulator.

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.
\* ===================================================================== */

#include "EmCommon.h"
#include "EmRegsCommandItf.h"

#include "EmBankRegs.h"			// RegsBank::GetROMSize
#include "EmBankROM.h"			// ROMBank::IsPCInRAM
#include "EmHAL.h"
#include "Byteswapping.h"		// ByteswapWords
#include "DebugMgr.h"			// Debug::CheckStepSpy
#include "Profiling.h"			// WAITSTATES_USB
#include "EmMemory.h"			// gMemoryAccess
#include "SessionFile.h"		// WriteUSBSonyRegsType, ReadUSBSonyRegsType, etc.


#define WAITSTATES_USB 0

typedef uint32 	(*ReadFunction)(emuptr address, int size);
typedef void	(*WriteFunction)(emuptr address, int size, uint32 value);


//*******************************************************************
// EmRegsCommandItf Class
//*******************************************************************

// Macro to return the Dragonball address of the specified register

#define addressof(x)	(this->GetAddressStart () + offsetof(HwrCommandItfType, x))

#define INSTALL_HANDLER(read, write, reg)			\
					this->SetHandler ((ReadFunction) &EmRegsCommandItf::read,		\
									  (WriteFunction) &EmRegsCommandItf::write,		\
									  addressof (reg),			\
									  sizeof (fRegs.reg))
#pragma mark -

// ---------------------------------------------------------------------------
//		 EmRegsCommandItf::EmRegsCommandItf
// ---------------------------------------------------------------------------

EmRegsCommandItf::EmRegsCommandItf ()
{
}

EmRegsCommandItf::EmRegsCommandItf (emuptr baseAddr) :
	fBaseAddr (baseAddr)
{
}


// ---------------------------------------------------------------------------
//		 EmRegsCommandItf::~EmRegsCommandItf
// ---------------------------------------------------------------------------

EmRegsCommandItf::~EmRegsCommandItf (void)
{
}

/***********************************************************************
 *
 * FUNCTION:	EmRegsCommandItf::Initialize
 *
 * DESCRIPTION: Standard initialization function.  Responsible for
 *				initializing this sub-system when a new session is
 *				created.  May also be called from the Load function
 *				to share common functionality.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void EmRegsCommandItf::Initialize (void)
{
	EmRegs::Initialize ();
}


/***********************************************************************
 *
 * FUNCTION:	EmRegsCommandItf::Reset
 *
 * DESCRIPTION: Standard reset function.  Sets the sub-system to a
 *				default state.	This occurs not only on a Reset (as
 *				from the menu item), but also when the sub-system
 *				is first initialized (Reset is called after Initialize)
 *				as well as when the system is re-loaded from an
 *				insufficient session file.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void EmRegsCommandItf::Reset (Bool hardwareReset)
{
	EmRegs::Reset (hardwareReset);
	if (hardwareReset)
	{
		memset (&fRegs, 0, sizeof(fRegs));
		fRegs.CmdItf0C06 = 0xFC;
	}
}


/***********************************************************************
 *
 * FUNCTION:	EmRegsCommandItf::Save
 *
 * DESCRIPTION: Standard save function.  Saves any sub-system state to
 *				the given session file.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void EmRegsCommandItf::Save (SessionFile& f)
{
	EmRegs::Save (f);
}


/***********************************************************************
 *
 * FUNCTION:	EmRegsCommandItf::Load
 *
 * DESCRIPTION: Standard load function.  Loads any sub-system state
 *				from the given session file.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void EmRegsCommandItf::Load (SessionFile& f)
{
	EmRegs::Load (f);
}


/***********************************************************************
 *
 * FUNCTION:	EmRegsCommandItf::Dispose
 *
 * DESCRIPTION: Standard dispose function.	Completely release any
 *				resources acquired or allocated in Initialize and/or
 *				Load.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void EmRegsCommandItf::Dispose (void)
{
}

// ---------------------------------------------------------------------------
//		 EmRegsCommandItf::SetSubBankHandlers
// ---------------------------------------------------------------------------
void EmRegsCommandItf::SetSubBankHandlers (void)
{
	// Install base handlers.

	EmRegs::SetSubBankHandlers ();

	// Now add standard/specialized handers for the defined registers.

	INSTALL_HANDLER (Read,	Write,		_filler01);
	INSTALL_HANDLER (Read,	Write,		CmdItf0C06);
	INSTALL_HANDLER (Read,	Write,		_filler02);
}

// ---------------------------------------------------------------------------
//		 EmRegsCommandItf::GetRealAddress
// ---------------------------------------------------------------------------

uint8* EmRegsCommandItf::GetRealAddress (emuptr address)
{
	uint8*	loc = ((uint8*) &fRegs) + (address - this->GetAddressStart ());
	return loc;
}

// ---------------------------------------------------------------------------
//		 EmRegsCommandItf::GetAddressStart
// ---------------------------------------------------------------------------

emuptr EmRegsCommandItf::GetAddressStart (void)
{
	return fBaseAddr; 
}


// ---------------------------------------------------------------------------
//		 EmRegsCommandItf::GetAddressRange
// ---------------------------------------------------------------------------

uint32 EmRegsCommandItf::GetAddressRange (void)
{
	return sizeof(fRegs);
}


// ---------------------------------------------------------------------------
//		 EmRegsCommandItf::Write
// ---------------------------------------------------------------------------
void EmRegsCommandItf::Write(emuptr address, int size, uint32 value)
{
	if (address == 0x11000C06 && size == 2)
	{
		if (this->StdReadBE (address, size) == 0xFC00)
			value = 0x0001;
	}

	this->StdWriteBE (address, size, value);
}

// ---------------------------------------------------------------------------
//		 EmRegsCommandItf::Read
// ---------------------------------------------------------------------------
uint32 EmRegsCommandItf::Read(emuptr address, int size)
{
	uint32	rstValue = this->StdReadBE (address, size);
	return	rstValue;
}

// ---------------------------------------------------------------------------
//		 EmRegsCommandItf::WrireToDummy
// ---------------------------------------------------------------------------
void EmRegsCommandItf::WrireToDummy (emuptr address, int size, uint32 value)
{
	UNUSED_PARAM(address)
	UNUSED_PARAM(size)

//	this->StdWriteBE (address, size, value);
}


// ---------------------------------------------------------------------------
//		 EmRegsCommandItf::ReadFromDummy
// ---------------------------------------------------------------------------

uint32 EmRegsCommandItf::ReadFromDummy (emuptr address, int size)
{
	UNUSED_PARAM(address)
	UNUSED_PARAM(size)

	return	this->StdReadBE (address, size);
}

// ---------------------------------------------------------------------------
//		 EmRegsCommandItf::ValidAddress
// ---------------------------------------------------------------------------

int EmRegsCommandItf::ValidAddress (emuptr address, uint32 iSize)
{
	UNUSED_PARAM(iSize)
	return 1;
}


// ---------------------------------------------------------------------------
//		 EmRegsCommandItf::InvalidAccess
// ---------------------------------------------------------------------------

void EmRegsCommandItf::InvalidAccess (emuptr address, long size, Bool forRead)
{
	EmAssert (gCPU68K);
	gCPU68K->BusError (address, size, forRead);
}


// ---------------------------------------------------------------------------
//		 EmRegsCommandItf::AddressError
// ---------------------------------------------------------------------------

void EmRegsCommandItf::AddressError (emuptr address, long size, Bool forRead)
{
	EmAssert (gCPU68K);
	gCPU68K->AddressError (address, size, forRead);
}

// ---------------------------------------------------------------------------
//		 USBSonyReg::UnsupportedWrite
// ---------------------------------------------------------------------------

void EmRegsCommandItf::UnsupportedWrite (emuptr address, int size, uint32 value)
{
	UNUSED_PARAM(value)

#ifdef SONY_ROM
	return ;
#endif

	if (!CEnableFullAccess::AccessOK ())
	{
		EmRegsCommandItf::InvalidAccess (address, size, false);
	}
}

#pragma mark -

